/*

 Security Software Testing Suite - Shared code library
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/

*/

#include <windows.h>
#include <stdio.h>
#include <tlhelp32.h>
#include <lmcons.h>
#include <ddk/ntifs.h>
#include <ddk/ntstatus.h>
#include "base64.h"
#include "common.h"

void com_verbosity_set(int value);
int com_verbosity_get(void);
void com_unhook_set(int value);
int com_unhook_get(void);
int com_console_init(char *name,char* file,char *line,PCOM_CONF conf,int type);
int com_console_init_winmain(char *name,char* file,char *line,PCOM_CONF conf,int type);
int com_console_init_winmain_gui(char *name,char* file,char *line,PCOM_CONF conf,int type);
int com_console_init_main(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type);
int com_console_init_main_no_about(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type);
int com_console_init_main_with_credits(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type,char *credits);
int com_target_info_load(char *name,PCOM_CONF conf,int type);

void com_about(char *name);
void com_usage(char *name);
void com_print_last_error(void);
int com_end_pref(int res,PCOM_ERROR err_inf,PCOM_CONF conf,char *prefix);
int com_end(int res,PCOM_ERROR err_inf,PCOM_CONF conf);

void com_err_set(PCOM_ERROR err_inf,const char *format,...);
void com_err_set_nc(PCOM_ERROR err_inf,const char *format,...);
void com_err_set_sc(PCOM_ERROR err_inf,DWORD code,const char *format,...);
void com_err_print(PCOM_ERROR err_inf,char *prefix);
void com_err_print_clear(PCOM_ERROR err_inf,char *prefix);
void com_err_set_print_clear(PCOM_ERROR err_inf,char *prefix,const char *format,...);

int com_get_image_info(char *path,PCOM_IMAGE_INFO info,PCOM_ERROR err_inf);

int com_mapping_server_create(PCOM_MAPPING map,size_t size,char *name,void *idata,size_t idata_size,PCOM_ERROR err_inf);
int com_mapping_client_open(PCOM_MAPPING map,size_t size,char *name,void *idata,size_t idata_size,PCOM_ERROR err_inf);
void com_mapping_close(PCOM_MAPPING mapping);

int com_get_windows_path(char *path,size_t size,PCOM_ERROR err_inf);
int com_get_system_path(char *path,size_t size,PCOM_ERROR err_inf);
int com_get_drivers_path(char *path,size_t size,PCOM_ERROR err_inf);
int com_get_module_name(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf);
int com_get_module_path(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf);
int com_get_module_image_name(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf);

int com_get_pid_from_name(ULONG *pid,char* name,PCOM_ERROR err_inf);
int com_get_name_from_pid(ULONG pid,char* name,size_t size,PCOM_ERROR err_inf);
int com_get_tids_from_pid(ULONG pid,ULONG *tids,int *tids_cnt,PCOM_ERROR err_inf);
int com_get_parent_info(ULONG pid,ULONG *ppid,char *name,size_t size,PCOM_ERROR err_inf);
int com_get_process_windows(ULONG pid,HWND *windows,int *windows_cnt,PCOM_ERROR err_inf);
int com_process_open(ULONG pid,DWORD access,int alternative,int verbose,HANDLE *handle,PCOM_ERROR err_inf);
int com_thread_open(ULONG tid,DWORD access,int alternative,int verbose,HANDLE *handle,PCOM_ERROR err_inf);
int com_proc_get_pids_from_names(char (*processes)[MAX_PATH],ULONG *pids,int cnt,PCOM_ERROR err_inf);
int com_proc_termination_report(char (*processes)[MAX_PATH],ULONG *pids,int cnt,PCOM_ERROR err_inf);

int com_find_pattern_and_print_data(char *data,size_t size,PCOM_CONF conf,PCOM_ERROR err_inf);

int com_security_info_set(HANDLE object,SE_OBJECT_TYPE type,SECURITY_INFORMATION info,DWORD access,ACCESS_MODE acmode,char *user,int merge,PCOM_ERROR err_inf);
int com_privilege_enable(char *priv_name,PCOM_ERROR err_inf);
int com_privilege_enable_debug(PCOM_ERROR err_inf);
int com_privilege_enable_shutdown(PCOM_ERROR err_inf);
int com_privilege_enable_backup(PCOM_ERROR err_inf);
int com_privilege_enable_restore(PCOM_ERROR err_inf);

int com_reg_key_copy(HKEY src_root,char *src_subkey,HKEY dst_root,char *dst_subkey);
int com_reg_root_name_to_hkey(char *root_name,HKEY *hkey,PCOM_ERROR err_inf);

int com_service_disable(char *svc_name,PCOM_ERROR err_inf);
int com_get_system_info_table(DWORD iclass,void **table,PCOM_ERROR err_inf);


// verbosity:
//  TRUE = be verbose on output
//  FALSE = do not be verbose
int verbose=FALSE;

// unhook:
//  TRUE = unhook important DLL functions
//  FALSE = do not unhook
int unhook=TRUE;


// console output and error files
FILE *conout=NULL;
FILE *conerr=NULL;

// console output changed
//  TRUE = console output and error output redirected to file
//  FALSE = standard console output and standard error output used
int conout_changed=FALSE;


/*
 Changes the value of the global 'verbose' variable.

 'value' A new value for the global 'verbose' variable.
*/

void com_verbosity_set(int value)
{
  verbose=value;
  return;
}


/*
 Obtains value of the global 'verbose' variable.

 Returns the value of the global 'verbose' variable.
*/

int com_verbosity_get(void)
{
  return verbose;
}


/*
 Changes the value of the global 'unhook' variable.

 'value' A new value for the global 'unhook' variable.
*/

void com_unhook_set(int value)
{
  unhook=value;
  return;
}


/*
 Obtains value of the global 'unhook' variable.

 Returns the value of the global 'unhook' variable.
*/

int com_unhook_get(void)
{
  return unhook;
}


/*
 Initialization routine for every console test.

 This function displays common about information on the standard output,
 parses the command line and loads the configuration.

 'name' A name of the test.
 'file' A name of the test's main module.
 'line' Command line parameter string similar to WinMain()'s 'lpCmdLine' parameter.
 'conf' A pointer to a structure to which the configuration is loaded.
 'type' Specifies the type of the test, it is one of COM_TEST_TYPE_* values.

 If the function succeeds the return value is TRUE, otherwise it is FALSE
 and the function displays common usage on the standard error output.
*/

int com_console_init(char *name,char* file,char *line,PCOM_CONF conf,int type)
{
  int opt_verbose=FALSE;
  int opt_unhook=TRUE;
  int opt_defaults=FALSE;

  int cmd_ok=TRUE;
  int clen=strlen(line);
  if (clen)
  {
    cmd_ok=FALSE;
    if (line[0]=='-')
    {
      cmd_ok=TRUE;
      line++;
      while (cmd_ok && *line)
      {
        switch (*line)
        {
          case 'd':
            if (!opt_defaults) opt_defaults=TRUE;
            else cmd_ok=FALSE;
            break;

          case 'v':
            if (!opt_verbose) opt_verbose=TRUE;
            else cmd_ok=FALSE;
            break;

          case 'u':
            if (opt_unhook) opt_unhook=FALSE;
            else cmd_ok=FALSE;
            break;

          default:
            cmd_ok=FALSE;
        }
        line++;
      }
    }
  }

  if (!cmd_ok)
  {
    com_usage(file);
    return FALSE;
  }

  com_verbosity_set(opt_verbose);
  com_unhook_set(opt_unhook);

  memset(conf,0,sizeof(conf));
  int res=opt_defaults;
  if (res)
  {
    if (type & COM_TEST_TYPE_USE_URL) lstrcpynA(conf->url,COM_CONF_DEF_URL,sizeof(conf->url));
    if (type & COM_TEST_TYPE_USE_PAGE) lstrcpynA(conf->url,COM_CONF_DEF_PAGE,sizeof(conf->url));

    if (type & COM_TEST_TYPE_USE_DATA_DELIM) lstrcpynA(conf->delim,COM_CONF_DEF_DATA_DELIM,sizeof(conf->delim));

    if (type & COM_TEST_TYPE_USE_DATA) snprintf(conf->data,COM_INET_DATA_MAX_LEN,"%s:%s",name,COM_CONF_DEF_DATA);
    if (type & COM_TEST_TYPE_USE_DATA46) snprintf(conf->data,COM_INET_DATA_MAX_LEN,"%s:%s",name,COM_CONF_DEF_DATA46);
    conf->data[COM_INET_DATA_MAX_LEN-1]='\0';

    if (type & COM_TEST_TYPE_USE_PATTERN_NET) lstrcpynA(conf->magic,COM_CONF_DEF_PATTERN_NET,sizeof(conf->magic));
    if (type & COM_TEST_TYPE_USE_PATTERN_SNIFF) lstrcpynA(conf->magic,COM_CONF_DEF_PATTERN_SNIFF,sizeof(conf->magic));

    if (type & COM_TEST_TYPE_USE_DOMAIN) lstrcpynA(conf->host,COM_CONF_DEF_DOMAIN,sizeof(conf->host));
    if (type & COM_TEST_TYPE_USE_DOMAIN2ND) lstrcpynA(conf->host,COM_CONF_DEF_DOMAIN2ND,sizeof(conf->host));

    if (type & COM_TEST_TYPE_USE_IP_TCP) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_IP_TCP);
    if (type & COM_TEST_TYPE_USE_IP_UDP) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_IP_UDP);
    if (type & COM_TEST_TYPE_USE_IP_RAW) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_IP_RAW);
    if (type & COM_TEST_TYPE_USE_IP_LOC) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_IP_LOC);
    if (type & COM_TEST_TYPE_USE_PEER_IP_TCP) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_PEER_IP_TCP);
    if (type & COM_TEST_TYPE_USE_PEER_IP_UDP) conf->ip.S_un.S_addr=inet_addr(COM_CONF_DEF_PEER_IP_UDP);

    if (type & COM_TEST_TYPE_USE_PORT_TCP) conf->port=COM_CONF_DEF_PORT_TCP;
    if (type & COM_TEST_TYPE_USE_PORT_UDP) conf->port=COM_CONF_DEF_PORT_UDP;
    if (type & COM_TEST_TYPE_USE_PEER_PORT_TCP) conf->port=COM_CONF_DEF_PEER_PORT_TCP;
    if (type & COM_TEST_TYPE_USE_PEER_PORT_UDP) conf->port=COM_CONF_DEF_PEER_PORT_UDP;

    if (type & COM_TEST_TYPE_USE_PEER_BUF_SIZE_TCP) conf->size=COM_CONF_DEF_PEER_BUF_SIZE_TCP;
    if (type & COM_TEST_TYPE_USE_PEER_BUF_SIZE_UDP) conf->size=COM_CONF_DEF_PEER_BUF_SIZE_UDP;

    if (type & COM_TEST_TYPE_USE_PEER_BUF_CNT_TCP) conf->count=COM_CONF_DEF_PEER_BUF_CNT_TCP;
    if (type & COM_TEST_TYPE_USE_PEER_BUF_CNT_UDP) conf->count=COM_CONF_DEF_PEER_BUF_CNT_UDP;
  } else res=com_target_info_load(name,conf,type);

  if (res)
  {
    res=base64_encode(conf->data,strlen(conf->data),conf->data64,sizeof(conf->data64))!=-1;
    if (res)
    {
      snprintf(conf->uri,sizeof(conf->uri),"%s%s%s",conf->url,conf->delim,conf->data64);
      conf->uri[sizeof(conf->uri)-1]='\0';

      if (com_verbosity_get())
      {
        fprintf(conout,"The following configuration will be used:\n");

        if (type & COM_TEST_TYPE_USE_URL) fprintf(conout,"  URL            = \"%s\"\n",conf->url);
        if (type & COM_TEST_TYPE_USE_PAGE) fprintf(conout,"  PAGE           = \"%s\"\n",conf->url);

        if (type & COM_TEST_TYPE_USE_DATA_DELIM) fprintf(conout,"  Delimiter      = \"%s\"\n",conf->delim);

        if (type & (COM_TEST_TYPE_USE_IP_TCP | COM_TEST_TYPE_USE_PEER_IP_TCP))
          fprintf(conout,"  Protocol       = TCP\n");

        if (type & (COM_TEST_TYPE_USE_IP_UDP | COM_TEST_TYPE_USE_PEER_IP_UDP))
          fprintf(conout,"  Protocol       = UDP\n");

        if (type & COM_TEST_TYPE_USE_IP_RAW)
          fprintf(conout,"  Protocol       = IP\n");

        if (type & (COM_TEST_TYPE_USE_IP_TCP | COM_TEST_TYPE_USE_IP_UDP | COM_TEST_TYPE_USE_IP_RAW
                  | COM_TEST_TYPE_USE_IP_LOC | COM_TEST_TYPE_USE_PEER_IP_TCP | COM_TEST_TYPE_USE_PEER_IP_UDP))
          fprintf(conout,"  IP address     = \"%s\"\n",inet_ntoa(conf->ip));

        if (type & (COM_TEST_TYPE_USE_PORT_TCP | COM_TEST_TYPE_USE_PORT_UDP
                  | COM_TEST_TYPE_USE_PEER_PORT_TCP | COM_TEST_TYPE_USE_PEER_PORT_UDP))
          fprintf(conout,"  Port           = %d\n",conf->port);

        if (type & (COM_TEST_TYPE_USE_DATA | COM_TEST_TYPE_USE_DATA46))
          fprintf(conout,"  Data           = \"%s\"\n"
                         "  Data (encoded) = \"%s\"\n",
                         conf->data,conf->data64);

        if (type & (COM_TEST_TYPE_USE_PATTERN_NET | COM_TEST_TYPE_USE_PATTERN_SNIFF))
          fprintf(conout,"  Magic pattern  = \"%s\"\n",conf->magic);

        if (type & (COM_TEST_TYPE_USE_DOMAIN | COM_TEST_TYPE_USE_DOMAIN2ND))
          fprintf(conout,"  Host           = \"%s\"\n",conf->host);

        if (type & (COM_TEST_TYPE_USE_PEER_BUF_SIZE_TCP | COM_TEST_TYPE_USE_PEER_BUF_SIZE_UDP))
          fprintf(conout,"  Packet size    = %d\n",conf->size);

        if (type & (COM_TEST_TYPE_USE_PEER_BUF_CNT_TCP | COM_TEST_TYPE_USE_PEER_BUF_CNT_UDP))
          fprintf(conout,"  Number of packets = %d\n",conf->count);

        fprintf(conout,"\n");

        if (type & COM_TEST_TYPE_USE_PROCESS_LIST)
        {
          fprintf(conout,"  Processes: ");
          if (conf->processes_cnt)
          {
            for (int i=0;i<conf->processes_cnt-1;i++)
              fprintf(conout,"%s, ",conf->processes[i]);

            fprintf(conout,"%s\n",conf->processes[conf->processes_cnt-1]);
          } else fprintf(conout,"NO PROCESSES\n");
        }

        if (type & COM_TEST_TYPE_USE_SERVICE_LIST)
        {
          fprintf(conout,"  Services: ");
          if (conf->services_cnt)
          {
            for (int i=0;i<conf->services_cnt-1;i++)
              fprintf(conout,"%s, ",conf->services[i]);

            fprintf(conout,"%s\n\n",conf->services[conf->services_cnt-1]);
          } else fprintf(conout,"NO SERVICES\n\n");
        }
      }
    } else fprintf(conerr,"Unable to encode user data.\n");
  }

  return res;
}


/*
 Initialization for every console test that implements WinMain().
 This function calls com_console_init().

 'name' A name of the test.
 'file' A name of the test's main module.
 'line' Forwarded WinMain()'s 'lpCmdLine' parameter.
 'conf' A pointer to a structure to which the configuration is loaded.
 'type' Specifies the type of the test, it is one of COM_TEST_TYPE_* values.

 If the function succeeds the return value is TRUE, otherwise it is FALSE
 and the function displays common usage on the standard error output.
*/

int com_console_init_winmain(char *name,char* file,char *line,PCOM_CONF conf,int type)
{
  conout=stdout;
  conerr=stderr;
  com_about(name);
  return com_console_init(name,file,line,conf,type);
}


/*
 Redirects standard output and standard error output to a text file
 and calls com_console_init() just like com_console_init_winmain().
 For arguments' description see com_console_init_winmain().
*/

int com_console_init_winmain_gui(char *name,char* file,char *line,PCOM_CONF conf,int type)
{
  char log_name[MAX_PATH];
  snprintf(log_name,sizeof(log_name),"%s.txt",file);
  log_name[sizeof(log_name)-1]='\0';

  conout=fopen(log_name,"w");
  if (!conout) return FALSE;

  conerr=conout;
  conout_changed=TRUE;

  com_about(name);
  return com_console_init(name,file,line,conf,type);
}


/*
 Initialization for every console test that implements main().
 This function calls com_console_init().

 'name' A name of the test.
 'file' A name of the test's main module.
 'argc' Forwarded main()'s 'argc' parameter.
 'argv' Forwarded main()'s 'argv' parameter.
 'conf' A pointer to a structure to which the configuration is loaded.
 'type' Specifies the type of the test, it is one of COM_TEST_TYPE_* values.

 If the function succeeds the return value is TRUE, otherwise it is FALSE
 and the function displays common usage on the standard error output.
*/

int com_console_init_main(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type)
{
  conout=stdout;
  conerr=stderr;
  com_about(name);
  if ((argc!=1) && (argc!=2))
  {
    com_usage(file);
    return FALSE;
  }
  return com_console_init(name,file,argc==2 ? argv[1] : "",conf,type);
}


/*
 The same as com_console_init_main() but does not print about information.
*/

int com_console_init_main_no_about(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type)
{
  conout=stdout;
  conerr=stderr;
  if ((argc!=1) && (argc!=2))
  {
    com_usage(file);
    return FALSE;
  }
  return com_console_init(name,file,argc==2 ? argv[1] : "",conf,type);
}


/*
 The same as com_console_init_main() but print credits information.

 'credits' Credits information.
*/

int com_console_init_main_with_credits(char *name,char* file,int argc,char **argv,PCOM_CONF conf,int type,char *credits)
{
  conout=stdout;
  conerr=stderr;
  com_about(name);

  fprintf(conout,"\nCredits:\n\n%s\n\n",credits);

  if ((argc!=1) && (argc!=2))
  {
    com_usage(file);
    return FALSE;
  }
  return com_console_init(name,file,argc==2 ? argv[1] : "",conf,type);
}


/*
 Loads configuration data from COM_CONFIG_FILE file.

 'name' A name of the test.
 'conf' A pointer to a structure to which the configuration is loaded.
 'type' Specifies the type of the test, it is one of COM_TEST_TYPE_*.

 If the function succeeds the return value is TRUE, otherwise the return value is FALSE.
*/

int com_target_info_load(char *name,PCOM_CONF conf,int type)
{
  int res=FALSE;

  FILE *fp=fopen(COM_CONFIG_FILE,"r");
  if (fp)
  {
    int agreement=FALSE;
    int line_cnt=0;
    char line[COM_CONFIG_FILE_MAX_LINE_LEN];

    int read_process_names=0,read_service_names=0;

    int error=FALSE;
    while (!feof(fp) && !error)
    {
       // skip comments and empty lines
      if (!fgets(line,sizeof(line),fp)) break;
      line_cnt++;

      if (line[0]=='#') continue;
      int len=strlen(line);
      while ((len>0) && ((line[len-1]=='\r') || (line[len-1]=='\n'))) len--;
      line[len]='\0';
      if (!strlen(line)) continue;

      if (read_process_names>0)
      {
        int proc_idx=conf->processes_cnt-read_process_names;
        lstrcpynA(conf->processes[proc_idx],line,sizeof(conf->processes[proc_idx]));
        read_process_names--;
        continue;
      } else if (read_service_names>0)
      {
        int svc_idx=conf->services_cnt-read_service_names;
        lstrcpynA(conf->services[svc_idx],line,sizeof(conf->services[svc_idx]));
        read_service_names--;
        continue;
      }


      char *val=strchr(line,'=');
      if (!val)
      {
        error=TRUE;
        fprintf(conerr,"ERROR (%s:%d): '=' is missing.\n",COM_CONFIG_FILE,line_cnt);
        break;
      }
      *val='\0';
      val++;

      if (!strlen(name) || !strlen(val))
      {
        error=TRUE;
        fprintf(conerr,"ERROR (%s:%d): The variable's name and its value must not be empty.\n",COM_CONFIG_FILE,line_cnt);
        break;
      }

      unsigned int val_type=COM_TEST_TYPE_USE_UNKNOWN;
      for (int i=0;i<com_conf_var_type_map_count;i++)
      {
        if (!strcmp(line,com_conf_var_type_map[i].name))
        {
          val_type=com_conf_var_type_map[i].type;
          break;
        }
      }

      if ((val_type!=COM_TEST_TYPE_USE_PROCESS_LIST)
       && (val_type!=COM_TEST_TYPE_USE_SERVICE_LIST)
       && (val_type!=COM_TEST_TYPE_USE_UNKNOWN)
       && ((val_type & type)==0)) continue;

      switch (val_type)
      {
        case COM_TEST_TYPE_USE_URL:
        case COM_TEST_TYPE_USE_PAGE:
          lstrcpynA(conf->url,val,sizeof(conf->url));
          break;

        case COM_TEST_TYPE_USE_DATA_DELIM:
          lstrcpynA(conf->delim,val,sizeof(conf->delim));
          break;

        case COM_TEST_TYPE_USE_DATA:
          snprintf(conf->data,COM_INET_DATA_MAX_LEN,"%s:%s",name,val);
          conf->data[COM_INET_DATA_MAX_LEN-1]='\0';
          break;

        case COM_TEST_TYPE_USE_DATA46:
          lstrcpynA(conf->data,val,46);
          conf->data[COM_INET_DATA_MAX_LEN-1]='\0';
          break;

        case COM_TEST_TYPE_USE_PATTERN_NET:
        case COM_TEST_TYPE_USE_PATTERN_SNIFF:
          lstrcpynA(conf->magic,val,sizeof(conf->magic));
          break;

        case COM_TEST_TYPE_USE_DOMAIN:
        case COM_TEST_TYPE_USE_DOMAIN2ND:
          lstrcpynA(conf->host,val,sizeof(conf->host));
          break;

        case COM_TEST_TYPE_USE_IP_TCP:
        case COM_TEST_TYPE_USE_IP_UDP:
        case COM_TEST_TYPE_USE_IP_RAW:
        case COM_TEST_TYPE_USE_IP_LOC:
        case COM_TEST_TYPE_USE_PEER_IP_TCP:
        case COM_TEST_TYPE_USE_PEER_IP_UDP:
          conf->ip.S_un.S_addr=inet_addr(val);
          if (conf->ip.S_un.S_addr==INADDR_NONE)
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid IP address.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;

        case COM_TEST_TYPE_USE_PORT_TCP:
        case COM_TEST_TYPE_USE_PORT_UDP:
        case COM_TEST_TYPE_USE_PEER_PORT_TCP:
        case COM_TEST_TYPE_USE_PEER_PORT_UDP:
          conf->port=strtoul(val,NULL,10);
          if ((conf->port<=0) || (conf->port>65535))
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid port number.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;

        case COM_TEST_TYPE_USE_PEER_BUF_SIZE_TCP:
        case COM_TEST_TYPE_USE_PEER_BUF_SIZE_UDP:
          conf->size=strtoul(val,NULL,10);

          // integer alignment
          if (conf->size % sizeof(int))
            conf->size+=sizeof(int)-(conf->size % sizeof(int));

          if ((conf->size<=0)
           || ((val_type==COM_TEST_TYPE_USE_PEER_BUF_SIZE_UDP) && (conf->size>=COM_INET_UDP_MAX_LEN)))
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid buffer / packet size.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;

        case COM_TEST_TYPE_USE_PEER_BUF_CNT_TCP:
        case COM_TEST_TYPE_USE_PEER_BUF_CNT_UDP:
          conf->count=strtoul(val,NULL,10);
          if (conf->count<=0)
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid number of buffers / packets.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;

        case COM_TEST_TYPE_USE_PROCESS_LIST:
        {
          conf->processes_cnt=strtoul(val,NULL,10);
          if ((conf->processes_cnt>=0) && (conf->processes_cnt<COM_CONFIG_FILE_MAX_PROCESSES))
          {
            if (conf->processes) free(conf->processes);
            conf->processes=NULL;

            read_process_names=conf->processes_cnt;

            if (read_process_names>0)
            {
              size_t size=conf->processes_cnt*MAX_PATH;
              conf->processes=malloc(size);
              if (conf->processes)
              {
                memset(conf->processes,0,size);
              } else
              {
                fprintf(conerr,"ERROR: Unable to allocate %d bytes of memory.\n",size);
                error=TRUE;
              }
            }
          } else
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid number of process names.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;
        }

        case COM_TEST_TYPE_USE_SERVICE_LIST:
        {
          conf->services_cnt=strtoul(val,NULL,10);
          if ((conf->services_cnt>=0) && (conf->services_cnt<COM_CONFIG_FILE_MAX_SERVICES))
          {
            if (conf->services) free(conf->services);
            conf->services=NULL;

            read_service_names=conf->services_cnt;

            if (read_service_names>0)
            {
              size_t size=conf->services_cnt*MAX_PATH;
              conf->services=malloc(size);
              if (conf->services)
              {
                memset(conf->services,0,size);
              } else
              {
                fprintf(conerr,"ERROR: Unable to allocate %d bytes of memory.\n",size);
                error=TRUE;
              }
            }
          } else
          {
            fprintf(conerr,"ERROR (%s:%d): \"%s\" is not a valid number of service names.\n",COM_CONFIG_FILE,line_cnt,val);
            error=TRUE;
          }
          break;
        }

        case COM_TEST_TYPE_USE_AGREEMENT:
          if (!strcmp(val,"I agree")) agreement=TRUE;
          break;

        default:
          error=TRUE;
          fprintf(conerr,"ERROR (%s:%d): Unknown variable name '%s'.\n",COM_CONFIG_FILE,line_cnt,line);
      }
    }

    if (!error && !agreement)
    {
      error=TRUE;
      fprintf(conerr,"ERROR: You have to read the configuration file and modify it before you can use this suite.\n");
    }


    if (error)
    {
      if (conf->processes) free(conf->processes);
      if (conf->services) free(conf->services);
      conf->processes=NULL;
      conf->services=NULL;
    } else res=TRUE;

    fclose(fp);
  } else fprintf(conerr,"Unable to open file \"%s\" for reading.\n",COM_CONFIG_FILE);

  return res;
}


/*
 Displays common about message on the standard output.

 'name' A name of the test.
*/

void com_about(char *name)
{
  fprintf(conout,"Security Software Testing Suite - %s\n"
                 "Copyright by www.matousec.com, Different Internet Experience Ltd.\n"
                 "http://www.matousec.com/""\n\n",name);
  return;
}


/*
 Displays common usage information on the standard error output.

 'file' A name of the test's main module.
*/

void com_usage(char *file)
{
  fprintf(conerr,"Usage: %s [options]\n"
                 "Options:\n"
                 "  -d          use default (hardcoded) settings, do not read configuration file \"%s\" (optional)\n"
                 "  -u          do NOT unhook important DLL functions (optional)\n"
                 "  -v          be verbose on output (optional)\n",
          file,COM_CONFIG_FILE);
  return;
}


/*
 Formats and prints the error message for GetLastError() code on the standard error output.
*/

void com_print_last_error(void)
{
  LPTSTR buf;
  DWORD code=GetLastError();
  if (FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,NULL,code,0,(LPTSTR)&buf,0,NULL))
  {
    fprintf(conerr,"Error code: %ld\n",code);
    fprintf(conerr,"Error message: %s",buf);
    LocalFree(buf);
  } else fprintf(conerr,"Unable to format error message for code %ld.\n",code);
  return;
}


/*
 Prints the test result and an error message if an error occurred with the given error prefix.

 'res' Set to TRUE if the test succeeded, set to FALSE otherwise.
 'err_inf' A pointer to a filled error information structure.
 'conf' A pointer to a structure with the loaded configuration.
 'prefix' A string to be inserted in front of the error message itself.

 Returns 0 if 'res' is TRUE, 1 otherwise.
*/

int com_end_pref(int res,PCOM_ERROR err_inf,PCOM_CONF conf,char *prefix)
{
  if (conf->processes) free(conf->processes);
  if (conf->services) free(conf->services);
  conf->processes=NULL;
  conf->services=NULL;

  com_err_print(err_inf,prefix);

  if (res) fprintf(conout,COM_MSG_TEST_SUCCEEDED);
  else fprintf(conout,COM_MSG_TEST_FAILED);

  if (conout_changed) fclose(conout);

  return (res ? 0 : 1);
}


/*
 Prints the test result and an error message if an error occurred.

 'res' Set to TRUE if the test succeeded, set to FALSE otherwise.
 'err_inf' A pointer to a filled error information structure.
 'conf' A pointer to a structure with the loaded configuration.

 Returns 0 if 'res' is TRUE, 1 otherwise.
*/

int com_end(int res,PCOM_ERROR err_inf,PCOM_CONF conf)
{
  return com_end_pref(res,err_inf,conf,COM_ERR_STANDARD_PREFIX);
}


/*
 Fills an error information structure with the current thread's last error code and an error message.

 'err_inf' A pointer to the error information structure to be filled.
 'format' printf()-like format string.
*/

void com_err_set(PCOM_ERROR err_inf,const char *format,...)
{
  err_inf->occurred=TRUE;
  err_inf->code=GetLastError();
  memset(err_inf->msg,0,sizeof(err_inf->msg));

  va_list args;
  va_start(args,format);

  vsnprintf(err_inf->msg,sizeof(err_inf->msg),format,args);
  va_end(args);

  err_inf->msg[sizeof(err_inf->msg)-1]='\0';
  return;
}


/*
 Fills an error information structure with an error message.

 'err_inf' A pointer to the error information structure to be filled.
 'format' printf()-like format string.
*/

void com_err_set_nc(PCOM_ERROR err_inf,const char *format,...)
{
  err_inf->occurred=TRUE;
  err_inf->code=0;
  memset(err_inf->msg,0,sizeof(err_inf->msg));

  va_list args;
  va_start(args,format);

  vsnprintf(err_inf->msg,sizeof(err_inf->msg),format,args);
  va_end(args);

  err_inf->msg[sizeof(err_inf->msg)-1]='\0';
  return;
}


/*
 Fills an error information structure with an error code and an error message.

 'err_inf' A pointer to the error information structure to be filled.
 'code' The error an code to be set.
 'format' printf()-like format string.
*/

void com_err_set_sc(PCOM_ERROR err_inf,DWORD code,const char *format,...)
{
  err_inf->occurred=TRUE;
  err_inf->code=code;
  memset(err_inf->msg,0,sizeof(err_inf->msg));

  va_list args;
  va_start(args,format);

  vsnprintf(err_inf->msg,sizeof(err_inf->msg),format,args);
  va_end(args);

  err_inf->msg[sizeof(err_inf->msg)-1]='\0';
  return;
}


/*
 Prints an error information on the standard error output.

 'err_inf' A pointer to the structure that holds information about the error.
 'prefix' A prefix string that is to be printed before the error message.
*/

void com_err_print(PCOM_ERROR err_inf,char *prefix)
{
  if (err_inf->occurred)
  {
    fprintf(conerr,"%s%s",prefix,err_inf->msg);
    if (err_inf->code!=ERROR_SUCCESS)
    {
      SetLastError(err_inf->code);
      com_print_last_error();
    }
  }
  return;
}


/*
 Prints an error information on the standard error output and clears the error information structure.

 'err_inf' A pointer to the structure that holds information about the error.
 'prefix' A prefix string that is to be printed before the error message.
*/

void com_err_print_clear(PCOM_ERROR err_inf,char *prefix)
{
  com_err_print(err_inf,prefix);
  err_inf->occurred=FALSE;
  return;
}


/*
 Fills an error information structure with the current thread's last error code and an error message.
 Prints the error information on the standard error output and clears the error information structure.

 'err_inf' A pointer to the structure that holds information about the error.
 'prefix' A prefix string that is to be printed before the error message.
 'format' printf()-like format string.
*/

void com_err_set_print_clear(PCOM_ERROR err_inf,char *prefix,const char *format,...)
{
  err_inf->occurred=TRUE;
  err_inf->code=GetLastError();
  memset(err_inf->msg,0,sizeof(err_inf->msg));

  va_list args;
  va_start(args,format);

  vsnprintf(err_inf->msg,sizeof(err_inf->msg),format,args);
  va_end(args);

  err_inf->msg[sizeof(err_inf->msg)-1]='\0';

  com_err_print(err_inf,prefix);

  err_inf->occurred=FALSE;
  return;
}



/*
 Collects information about an executable file.

 'path' A full path to the executable.
 'info' Points to a caller allocated structure that is filled with the information about the executable.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_image_info(char *path,PCOM_IMAGE_INFO info,PCOM_ERROR err_inf)
{
  int res=FALSE;
  HANDLE file=CreateFile(path,GENERIC_READ,FILE_SHARE_READ | FILE_SHARE_WRITE,NULL,OPEN_EXISTING,0,NULL);
  if (file!=INVALID_HANDLE_VALUE)
  {
    char buffer[4096];

    DWORD bread;
    if (ReadFile(file,buffer,sizeof(buffer),&bread,NULL))
    {
      char *limit=buffer+bread;
      PIMAGE_DOS_HEADER dos_hdr=(PIMAGE_DOS_HEADER)buffer;
      int invalid_image=TRUE;

      if ((size_t)limit>(size_t)&dos_hdr->e_lfanew+sizeof(dos_hdr->e_lfanew))
      {
        PIMAGE_NT_HEADERS nt_hdr=(PIMAGE_NT_HEADERS)(buffer+dos_hdr->e_lfanew);
        if ((size_t)limit>(size_t)nt_hdr+sizeof(IMAGE_NT_HEADERS))
        {
          info->image_base=(PVOID)nt_hdr->OptionalHeader.ImageBase;
          info->image_size=nt_hdr->OptionalHeader.SizeOfImage;
          info->entry_point=(PVOID)((size_t)info->image_base+nt_hdr->OptionalHeader.AddressOfEntryPoint);
          info->sub_system=nt_hdr->OptionalHeader.Subsystem;
          invalid_image=FALSE;
          res=TRUE;
        }
      }

      if (invalid_image) fprintf(conerr,"\"%s\" is not a valid image.\n",path);
    } else com_err_set(err_inf,"Unable to read from file \"%s\".\n",path);

    CloseHandle(file);
  } else com_err_set(err_inf,"Unable to open file \"%s\" for reading.\n",path);
  return res;
}



/*
 Creates a file mapping and a synchronization event.
 The mapping should be closed using com_mapping_close() when it is no more needed.

 'mapping' A pointer to a structure that stores information about the mapping.
 'size' The mapping size.
 'name' A name of the test with an optional suffix if more mappings are needed.
 'idata' A pointer to the initial data to be stored in the mapping, this parameter can be NULL.
 'idata_size' The initial data size.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_mapping_server_create(PCOM_MAPPING map,size_t size,char *name,void *idata,size_t idata_size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  char evt_name[COM_EVENT_NAME_MAX_LEN];
  char map_name[COM_MAPPING_NAME_MAX_LEN];
  snprintf(evt_name,sizeof(evt_name),"%s%s%s",COM_EVENT_PREFIX,name,COM_EVENT_SUFFIX);
  evt_name[sizeof(evt_name)-1]='\0';
  snprintf(map_name,sizeof(map_name),"%s%s%s",COM_MAPPING_PREFIX,name,COM_MAPPING_SUFFIX);
  map_name[sizeof(map_name)-1]='\0';

  HANDLE event=CreateEvent(NULL,FALSE,FALSE,evt_name);
  if (event)
  {
    HANDLE mapping=CreateFileMapping(INVALID_HANDLE_VALUE,NULL,PAGE_READWRITE,0,size,map_name);
    if (mapping)
    {
      void *data=MapViewOfFile(mapping,FILE_MAP_READ | FILE_MAP_WRITE,0,0,size);
      if (data)
      {
        memset(data,0,size);
        map->event=event;
        map->mapping=mapping;
        map->data=data;
        if (idata) memcpy(map->data,idata,idata_size);

        res=TRUE;

        if (!res) UnmapViewOfFile(data);
      } else com_err_set(err_inf,"Unable to map view of file \"%s\" to current process.\n",map_name);

      if (!res) CloseHandle(mapping);
    } else com_err_set(err_inf,"Unable to create file mapping \"%s\".\n",map_name);

    if (!res) CloseHandle(event);
  } else com_err_set(err_inf,"Unable to create event \"%s\".\n",evt_name);

  return res;
}


/*
 Opens the mapping created via com_mapping_server_create().
 The mapping should be closed using com_mapping_close() when it is no more needed.

 'mapping' A pointer to a structure that stores information about the mapping.
 'size' The mapping size.
 'name' A name of the test with an optional suffix if more mappings are needed.
 'idata' A pointer to a buffer that is filled with the initial mapping data (see com_mapping_server_create()),
 this parameter can be NULL.
 'idata_size' The size of 'idata' buffer.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_mapping_client_open(PCOM_MAPPING map,size_t size,char *name,void *idata,size_t idata_size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  char evt_name[COM_EVENT_NAME_MAX_LEN];
  char map_name[COM_MAPPING_NAME_MAX_LEN];
  snprintf(evt_name,sizeof(evt_name),"%s%s%s",COM_EVENT_PREFIX,name,COM_EVENT_SUFFIX);
  evt_name[sizeof(evt_name)-1]='\0';
  snprintf(map_name,sizeof(map_name),"%s%s%s",COM_MAPPING_PREFIX,name,COM_MAPPING_SUFFIX);
  map_name[sizeof(map_name)-1]='\0';

  HANDLE event=OpenEvent(EVENT_ALL_ACCESS,FALSE,evt_name);
  if (event)
  {
    HANDLE mapping=OpenFileMapping(FILE_MAP_ALL_ACCESS,FALSE,map_name);
    if (mapping)
    {
      void *data=MapViewOfFile(mapping,FILE_MAP_READ | FILE_MAP_WRITE,0,0,size);
      if (data)
      {
        map->event=event;
        map->mapping=mapping;
        map->data=data;
        if (idata) memcpy(idata,map->data,idata_size);
        memset(data,0,size);

        res=TRUE;

        if (!res) UnmapViewOfFile(data);
      } else com_err_set(err_inf,"Unable to map view of file \"%s\" to current process.\n",map_name);

      if (!res) CloseHandle(mapping);
    } else com_err_set(err_inf,"Unable to open file mapping \"%s\".\n",map_name);

    if (!res) CloseHandle(event);
  } else com_err_set(err_inf,"Unable to open event \"%s\".\n",evt_name);

  return res;
}

/*
 Closes a file mapping and synchronization event created via com_mapping_server_create().

 'mapping' A pointer to a structure that stores information about the mapping.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

void com_mapping_close(PCOM_MAPPING mapping)
{
  UnmapViewOfFile(mapping->data);
  CloseHandle(mapping->mapping);
  CloseHandle(mapping->event);
  return;
}



/*
 Obtains path to WINDOWS directory.

 'path' A pointer to a buffer that will be filled with the Windows directory name. This path never ends with a backslash.
 'size' A size of 'path' buffer, should be set to MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_get_windows_path(char *path,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  int len=GetWindowsDirectory(path,size);
  if (len)
  {
    path[size-1]='\0';
    if (len<=size)
    {
      if (path[len-1]=='\\') path[len-1]='\0';
      res=TRUE;
    } else com_err_set_nc(err_inf,"Buffer size too small for the Windows directory path.\n");
  } else com_err_set(err_inf,"Unable to get the Windows directory.\n");

  return res;
}


/*
 Obtains path to system directory.

 'path' A pointer to a buffer that will be filled with the system directory name. This path never ends with a backslash.
 'size' A size of 'path' buffer, should be set to MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_get_system_path(char *path,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  int len=GetSystemDirectory(path,size);
  if (len)
  {
    path[size-1]='\0';
    if (len<=size)
    {
      if (path[len-1]=='\\') path[len-1]='\0';
      res=TRUE;
    } else com_err_set_nc(err_inf,"Buffer size too small for the system directory path.\n");
  } else com_err_set(err_inf,"Unable to get the system directory.\n");

  return res;
}


/*
 Obtains the system's drivers path.

 'path' points to a buffer that is filled with the system's drivers path. This path never ends with a backslash.
 'size' A size of 'path' buffer, should be set to MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_get_drivers_path(char *path,size_t size,PCOM_ERROR err_inf)
{
  int res=com_get_system_path(path,size,err_inf);

  if (res)
  {
    int len=strlen(path);
    res=strlen("\\drivers")<(size-len);
    if (res) lstrcpynA(path+len,"\\drivers",size-len);
    else com_err_set_nc(err_inf,"Buffer size too small for the system's driver directory path.\n");
  }

  return res;
}


/*
 Obtains a full name of the given module.

 'module' The module handle. If 'module' is NULL the function obtains name of the main process module.
 'name' A pointer to a buffer that will be filled with a full name of the given module.
 'size' A size of 'name' buffer, should be at least MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_get_module_name(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;
  DWORD len=GetModuleFileName(module,name,size);
  if (len)
  {
    name[size-1]='\0';
    res=TRUE;
  } else
  {
    if (module) com_err_set(err_inf,"Unable to get full name of the main module.\n");
    else com_err_set(err_inf,"Unable to get full name of the module loaded at 0x%p.\n",module);
  }

  return res;
}


/*
 Obtains a path to the given module.

 'module' The module handle. If 'module' is NULL the function obtains path of the main process module.
 This path never ends with a backslash.
 'path' A pointer to a buffer that will be filled with a full name of the given module.
 'size' A size of 'path' buffer, should be at least MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/


int com_get_module_path(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf)
{
  int res=com_get_module_name(module,name,size,err_inf);
  if (res)
  {
    char *sl=strrchr(name,'\\');
    if (sl) *sl='\0';
  } else
  {
    if (module) com_err_set(err_inf,"Unable to get full name of the main module.\n");
    else com_err_set(err_inf,"Unable to get full name of the module loaded at 0x%p.\n",module);
  }

  return res;
}



/*
 Obtains a name of the given module without a path.

 'module' The module handle. If 'module' is NULL the function obtains name of the main process module.
 'name' A pointer to a buffer that will be filled with a name of the given module.
 'size' A size of 'name' buffer, should be at least MAX_PATH.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_get_module_image_name(HMODULE module,char *name,size_t size,PCOM_ERROR err_inf)
{
  char full[MAX_PATH];
  int res=com_get_module_name(module,full,MAX_PATH,err_inf);
  if (res)
  {
    char *sl=strrchr(full,'\\');
    if (sl) sl++;
    else sl=full;

    lstrcpynA(name,sl,size);
  }

  return res;
}


/*
 Converts process name to process identifier. If more than one process of the given name run, the returned
 identifier belongs to one of them.

 'pid' A pointer to a variable that receives the identifier.
 'name' A pointer to a null-terminated name of the process.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 If no process of the given name runs, the return value is TRUE and 'pid' is set to COM_CID_INVALID.
 If the function fails 'pid' is set to COM_CID_INVALID.
*/

int com_get_pid_from_name(ULONG *pid,char* name,PCOM_ERROR err_inf)
{
  int res=FALSE;
  PROCESSENTRY32 pe;
  pe.dwSize=sizeof(pe);

  *pid=COM_CID_INVALID;
  HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (shot)
  {
    if (Process32First(shot,&pe))
    {
      do
      {
        if (!stricmp(name,pe.szExeFile))
        {
          *pid=pe.th32ProcessID;
          break;
        }
      } while (Process32Next(shot,&pe));
      res=TRUE;

    } else com_err_set(err_inf,"Unable to get the first process from the snapshot.\n");

    CloseHandle(shot);
  } else com_err_set(err_inf,"Unable to create toolhelp process snapshot.\n");

  return res;
}


/*
 Converts process identifier to process name.

 'pid' A process identifier.
 'name' A pointer to an array of chars that will be filled with a null-terminated name of the process.
 'size' A size of 'name' in chars.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 If no process of the given identifier runs, the return value is TRUE and first char of 'name' is set to '\0'.
*/

int com_get_name_from_pid(ULONG pid,char* name,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;
  name[0]='\0';

  PROCESSENTRY32 pe;
  pe.dwSize=sizeof(pe);

  HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (shot)
  {
    if (Process32First(shot,&pe))
    {
      do
      {
        if (pid==pe.th32ProcessID)
        {
          lstrcpynA(name,pe.szExeFile,size);
          break;
        }
      } while (Process32Next(shot,&pe));
      res=TRUE;

    } else com_err_set(err_inf,"Unable to get the first process from the snapshot.\n");

    CloseHandle(shot);
  } else com_err_set(err_inf,"Unable to create toolhelp process snapshot.\n");

  return res;
}


/*
 Obtains identifiers of threads that belong to a given process.

 'pid' A process identifier.
 'tids' A pointer to an array of ULONG variables that will be filled with thread identifiers.
 'tids_cnt' A pointer to a variable that is set to a number of fields in 'tids' array on the input.
 On the output, this variable will be set to the number of identifiers written to 'tids' array in case
 there are not as many threads in the target process to fill the whole array.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 The number of identifiers written to 'tids' array is returned via 'tids_cnt'.
 The function succeeds only if the number of identifiers written to 'tids' array is at least 1.
*/

int com_get_tids_from_pid(ULONG pid,ULONG *tids,int *tids_cnt,PCOM_ERROR err_inf)
{
  int res=FALSE;

  THREADENTRY32 te;
  te.dwSize=sizeof(te);

  int cnt=*tids_cnt;
  *tids_cnt=0;

  if (cnt)
  {
    HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPTHREAD,0);
    if (shot)
    {
      if (Thread32First(shot,&te))
      {
        do
        {
          if (te.th32OwnerProcessID==pid)
          {
            tids[*tids_cnt]=te.th32ThreadID;
            *tids_cnt=*tids_cnt+1;
          }
        } while (Thread32Next(shot,&te) && (*tids_cnt<cnt));

        res=(*tids_cnt>0);

      } else com_err_set(err_inf,"Unable to get the first thread from the snapshot.\n");

      CloseHandle(shot);
    } else com_err_set(err_inf,"Unable to create toolhelp thread snapshot.\n");
  }

  return res;
}


/*
 Obtains a parent process identifier and name of the selected process.

 'pid' A process identifier.
 'ppid' A pointer to a variable that receives the parent process identifier. This pointer can be NULL.
 'name' A pointer to an array of chars that will be filled with a null-terminated name of the parent process. This pointer can be NULL.
 'size' A size of 'name' in chars.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 If no process of the given identifier runs, the function fails.
 If the parent process does not exist any more and 'name' is not NULL, the first char of 'name' is set to '\0'.
*/

int com_get_parent_info(ULONG pid,ULONG *ppid,char *name,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  PROCESSENTRY32 pe;
  pe.dwSize=sizeof(pe);

  ULONG parent=COM_CID_INVALID;
  HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (shot)
  {
    if (Process32First(shot,&pe))
    {
      do
      {
        if (pid==pe.th32ProcessID)
        {
          parent=pe.th32ParentProcessID;
          if (ppid) *ppid=parent;
          res=TRUE;
          break;
        }
      } while (Process32Next(shot,&pe));
    } else com_err_set(err_inf,"Unable to get the first process from the snapshot (1).\n");

    if ((parent!=COM_CID_INVALID) && name)
    {
      name[0]='\0';
      if (Process32First(shot,&pe))
      {
        do
        {
          if (parent==pe.th32ProcessID)
          {
            lstrcpynA(name,pe.szExeFile,size);
            break;
          }
        } while (Process32Next(shot,&pe));
      } else com_err_set(err_inf,"Unable to get the first process from the snapshot (2).\n");
    }

    CloseHandle(shot);
  } else com_err_set(err_inf,"Unable to create toolhelp process snapshot.\n");

  return res;
}


/*
 Callback function used by com_get_process_windows().
 Collects handles to windows of the selected process.
 For more information see 'lpEnumFunc' parameter of EnumWindows API.
*/

BOOL CALLBACK com_get_process_windows_callback(HWND hwnd,LPARAM lParam)
{
  struct
  {
    ULONG pid;
    HWND *windows;
    int max_cnt;
    int *windows_cnt;
  } *cb_struct=(void*)lParam;

  DWORD pid;
  DWORD tid=GetWindowThreadProcessId(hwnd,&pid);
  if (tid && (pid==cb_struct->pid) && (*cb_struct->windows_cnt<cb_struct->max_cnt))
  {
    cb_struct->windows[*cb_struct->windows_cnt]=hwnd;
    *cb_struct->windows_cnt=*cb_struct->windows_cnt+1;
  }

  return TRUE;
}


/*
 Obtains handles of windows that belong to a given process.

 'pid' A process identifier.
 'windows' A pointer to an array of HWND variables that will be filled with handles of windows.
 'windows_cnt' A pointer to a variable that is set to a number of fields in 'windows' array on the input.
 On the output, this variable will be set to the number of handles written to 'windows' array in case
 there are not enough windows of the given process to fill the whole array.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 The number of handles written to 'windows' array is returned via 'windows_cnt'.
 The function succeeds even if no windows were found for the given process (i.e. 'windows_cnt'
 is set to 0 on the output).
*/


int com_get_process_windows(ULONG pid,HWND *windows,int *windows_cnt,PCOM_ERROR err_inf)
{
  int res=FALSE;

  struct
  {
    ULONG pid;
    HWND *windows;
    int max_cnt;
    int *windows_cnt;
  } cb_struct={pid,windows,*windows_cnt,windows_cnt};

  *windows_cnt=0;

  res=EnumWindows(com_get_process_windows_callback,(LPARAM)&cb_struct);
  if (!res) com_err_set(err_inf,"Unable to enumerate applications' windows.\n");

  return res;
}



/*
 Obtains a handle to a process.

 If the process can not be opened using OpenProcess API, we try two tricks to obtain it.
 Firstly, we try to open the process will SYNCHRONIZE access right only and then duplicate this handle
 in order to get more access rights. If we can not get a valid handle using this method, we try the handle
 stealing method to steal the handle from "csrss.exe". "csrss.exe" is a system process that owns handles to
 all processes in the system and these handles can be copied to (stolen by) another process and used as
 if they were obtained using OpenProcess API.

 This function does not adjust the caller thread's privileges. In case of system process or handle stealing method,
 the caller thread is expected to have Debug privilege.

 'pid' The target process identifier.
 'access' See 'dwDesiredAccess' parameter of OpenProcess API. If the handle is stolen from "csrss.exe",
 this parameter is ignored and the handle is copied with the access that the original handle of "csrss.exe" has.
 'alternative' If set to TRUE, then if the function fails to open the process directly using OpenProcess API,
 it attempts to obtain a handle using duplication trick or via the handle stealing method from "csrss.exe".
 'verbose' Use standard console output to inform the user that the function uses stealing method.
 'handle' A pointer to a variable that receives a handle to the process.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_process_open(ULONG pid,DWORD access,int alternative,int verbose,HANDLE *handle,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HANDLE proc=OpenProcess(access,FALSE,pid);
  if (proc)
  {
    *handle=proc;
    res=TRUE;
  } else if (alternative)
  {
    if (verbose) fprintf(conout,"Unable to open process PID %ld directly, trying duplication method ...\n",pid);
    proc=OpenProcess(SYNCHRONIZE,FALSE,pid);
    if (proc)
    {
      HANDLE proc_dup;
      if (DuplicateHandle(GetCurrentProcess(),proc,GetCurrentProcess(),&proc_dup,access,FALSE,0))
      {
        *handle=proc_dup;
        res=TRUE;
      }

      CloseHandle(proc);
    }

    if (!res)
    {
      if (verbose) fprintf(conout,"Unable to open process PID %ld using duplication method, trying handle stealing method ...\n",pid);

      ULONG curpid=GetCurrentProcessId();
      HANDLE curproc=OpenProcess(PROCESS_ALL_ACCESS,FALSE,curpid);
      if (curproc)
      {
        PSYSTEM_HANDLE_INFORMATION_BUFFER handle_table=NULL;
        if (com_get_system_info_table(SystemHandleInformation,(void*)&handle_table,err_inf))
        {
          UCHAR process_object_type;
          int process_object_type_found=FALSE;

          // find 'curproc' handle and thus get type of process handles
          for (int i=0;i<handle_table->HandleCount;i++)
          {
            if ((handle_table->HandleInfo[i].ProcessId==curpid)
             && (handle_table->HandleInfo[i].Handle==(USHORT)(ULONG)curproc))
            {
              process_object_type=handle_table->HandleInfo[i].ObjectTypeNumber;
              process_object_type_found=TRUE;
              break;
            }
          }

          if (process_object_type_found)
          {
            ULONG csrss_pid;
            int ret=com_get_pid_from_name(&csrss_pid,"csrss.exe",err_inf);
            if (ret && (csrss_pid!=COM_CID_INVALID))
            {
              HANDLE csrss=OpenProcess(PROCESS_DUP_HANDLE,FALSE,csrss_pid);
              if (csrss)
              {
                // find target process handle that belongs to "csrss.exe"
                int handle_idx=0;
                while (handle_idx<handle_table->HandleCount)
                {
                  if ((handle_table->HandleInfo[handle_idx].ProcessId==csrss_pid)
                   && (handle_table->HandleInfo[handle_idx].ObjectTypeNumber==process_object_type))
                  {
                    HANDLE process_handle=NULL;
                    if (DuplicateHandle(csrss,(HANDLE)(ULONG)handle_table->HandleInfo[handle_idx].Handle,GetCurrentProcess(),
                                        &process_handle,0,FALSE,DUPLICATE_SAME_ACCESS))
                    {
                      PROCESS_BASIC_INFORMATION process_info;
                      NTSTATUS status=ZwQueryInformationProcess(process_handle,ProcessBasicInformation,
                                                                &process_info,sizeof(process_info),NULL);
                      if (NT_SUCCESS(status))
                      {
                        // if the handle we duplicated was a handle of the target process
                        if (process_info.UniqueProcessId==pid)
                        {
                          *handle=process_handle;
                          res=TRUE;
                          break;
                        }
                      }

                      CloseHandle(process_handle);
                    }
                  }

                  handle_idx++;
                }

                if (!res) com_err_set_nc(err_inf,"Unable to find handle of the target process (PID %ld) among handles of \"csrss.exe\".\n",pid);

                CloseHandle(csrss);
              } else com_err_set(err_inf,"Unable to open process \"csrss.exe\" (PID %ld).\n",csrss_pid);
            } else if (ret) com_err_set(err_inf,"ERROR: Unable to find process \"csrss.exe\".\n");
          } else com_err_set_nc(err_inf,"Unable to determine the system object type of process objects.\n");

          free(handle_table);
        }

        CloseHandle(curproc);
      } else com_err_set(err_inf,"Unable to open current process.\n");
    }
  } else com_err_set(err_inf,"Unable to open process PID %ld.\n",pid);

  return res;
}


/*
 Obtains a handle to a thread.

 If the thraed can not be opened using OpenThread API, we try two tricks to obtain it.
 Firstly, we try to open the thread will SYNCHRONIZE access right only and then duplicate this handle
 in order to get more access rights. If we can not get a valid handle using this method, we try the handle
 stealing method to steal the handle from "csrss.exe". "csrss.exe" is a system process that owns handles to
 all threads in the system and these handles can be copied to (stolen by) another process and used as
 if they were obtained using OpenThread API.

 This function does not adjust the caller thread's privileges. In case of system process or handle stealing method,
 the caller thread is expected to have Debug privilege.

 'tid' The target thread identifier.
 'access' See 'dwDesiredAccess' parameter of OpenThread API. If the handle is stolen from "csrss.exe",
 this parameter is ignored and the handle is copied with the access that the original handle of "csrss.exe" has.
 'steal_handle' If set to TRUE, then if the function fails to open the process directly using OpenThread API,
 it attempts to obtain a handle of the target thread from "csrss.exe" using the handle stealing technique.
 'verbose' Use standard console output to inform the user that the function uses stealing method.
 'handle' A pointer to a variable that receives a handle to the thread.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int com_thread_open(ULONG tid,DWORD access,int alternative,int verbose,HANDLE *handle,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HANDLE thread=OpenThread(access,FALSE,tid);
  if (thread)
  {
    *handle=thread;
    res=TRUE;
  } else if (alternative)
  {
    if (verbose) fprintf(conout,"Unable to open thread TID %ld directly, trying duplication method ...\n",tid);

    thread=OpenThread(SYNCHRONIZE,FALSE,tid);
    if (thread)
    {
      HANDLE thread_dup;
      if (DuplicateHandle(GetCurrentProcess(),thread,GetCurrentProcess(),&thread_dup,access,FALSE,0))
      {
        *handle=thread_dup;
        res=TRUE;
      }

      CloseHandle(thread);
    }

    if (!res)
    {
      if (verbose) fprintf(conout,"Unable to open thread TID %ld using duplication method, trying handle stealing method ...\n",tid);

      ULONG curpid=GetCurrentProcessId();
      ULONG curtid=GetCurrentThreadId();
      HANDLE curthread=OpenThread(THREAD_ALL_ACCESS,FALSE,curtid);
      if (curthread)
      {
        PSYSTEM_HANDLE_INFORMATION_BUFFER handle_table=NULL;
        if (com_get_system_info_table(SystemHandleInformation,(void*)&handle_table,err_inf))
        {
          UCHAR thread_object_type;
          int thread_object_type_found=FALSE;

          // find 'curthread' handle and thus get type of thread handles
          for (int i=0;i<handle_table->HandleCount;i++)
          {
            if ((handle_table->HandleInfo[i].ProcessId==curpid)
             && (handle_table->HandleInfo[i].Handle==(USHORT)(ULONG)curthread))
            {
              thread_object_type=handle_table->HandleInfo[i].ObjectTypeNumber;
              thread_object_type_found=TRUE;
              break;
            }
          }

          if (thread_object_type_found)
          {
            ULONG csrss_pid;
            int ret=com_get_pid_from_name(&csrss_pid,"csrss.exe",err_inf);
            if (ret && (csrss_pid!=COM_CID_INVALID))
            {
              HANDLE csrss=OpenProcess(PROCESS_DUP_HANDLE,FALSE,csrss_pid);
              if (csrss)
              {
                // find target thread handle that belongs to "csrss.exe"
                int handle_idx=0;
                while (handle_idx<handle_table->HandleCount)
                {
                  if ((handle_table->HandleInfo[handle_idx].ProcessId==csrss_pid)
                   && (handle_table->HandleInfo[handle_idx].ObjectTypeNumber==thread_object_type))
                  {
                    HANDLE thread_handle=NULL;
                    if (DuplicateHandle(csrss,(HANDLE)(ULONG)handle_table->HandleInfo[handle_idx].Handle,GetCurrentProcess(),
                                        &thread_handle,0,FALSE,DUPLICATE_SAME_ACCESS))
                    {
                      THREAD_BASIC_INFORMATION thread_info;
                      NTSTATUS status=ZwQueryInformationThread(thread_handle,ThreadBasicInformation,
                                                                &thread_info,sizeof(thread_info),NULL);
                      if (NT_SUCCESS(status))
                      {
                        // if the handle we duplicated was a handle of the target process
                        if ((ULONG)thread_info.ClientId.UniqueThread==tid)
                        {
                          *handle=thread_handle;
                          res=TRUE;
                          break;
                        }
                      }

                      CloseHandle(thread_handle);
                    }
                  }

                  handle_idx++;
                }

                if (!res) com_err_set_nc(err_inf,"Unable to find handle of the target thread (TID %ld) among handles of \"csrss.exe\".\n",tid);

                CloseHandle(csrss);
              } else com_err_set(err_inf,"Unable to open process \"csrss.exe\" (PID %ld).\n",csrss_pid);
            } else if (ret) com_err_set(err_inf,"ERROR: Unable to find process \"csrss.exe\".\n");
          } else com_err_set_nc(err_inf,"Unable to determine the system object type of process objects.\n");

          free(handle_table);
        }

        CloseHandle(curthread);
      } else com_err_set(err_inf,"Unable to open current process.\n");
    }
  } else com_err_set(err_inf,"Unable to open thread TID %ld.\n",tid);

  return res;
}



/*
 Converts a list of process names to a list of process identifiers.
 Unlike com_get_pid_from_name(), this function can be used to get more than one identifier
 of processes of the same name.

 'processes' A pointer to an array of 'cnt' process names.
 'pids' A pointer to an array that receives 'cnt' process identifiers. On the output, pids[i] contains
 the identifier of the process of the name processes[i]. If such process does not exist, pids[i] is set
 to COM_CID_INVALID.
 'cnt' A number of process names in 'processes' array and the minimal
 number of slots for identifiers in 'pids' array.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
 The function succeeds if at least one process identifier has been written to 'pids' array.
*/

int com_proc_get_pids_from_names(char (*processes)[MAX_PATH],ULONG *pids,int cnt,PCOM_ERROR err_inf)
{
  int res=FALSE;

  PROCESSENTRY32 pe;
  pe.dwSize=sizeof(pe);

  for (int i=0;i<cnt;i++) pids[i]=COM_CID_INVALID;

  HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (shot)
  {
    if (Process32First(shot,&pe))
    {
      do
      {
        for (int i=0;i<cnt;i++)
        {
          if ((pids[i]==COM_CID_INVALID) && !stricmp(processes[i],pe.szExeFile))
          {
            pids[i]=pe.th32ProcessID;
            res=TRUE;
            break;
          }
        }
      } while (Process32Next(shot,&pe));

    } else com_err_set(err_inf,"Unable to get the first process from the snapshot.\n");

    CloseHandle(shot);
  } else com_err_set(err_inf,"Unable to create toolhelp process snapshot.\n");

  return res;
}


/*
 Repeatedly checks whether the processes of the given identifiers still exist by looking up their names
 in the list of running processes. The function might report wrong results in case the process was terminated
 and spawned again and received the same identifier as before.

 'processes' A pointer to an array of 'cnt' process names.
 'pids' A pointer to an array of 'cnt' process identifiers that correspond to the process names given in 'processes'.
 This array may contain COM_CID_INVALID values, such processes are not checked.
 'cnt' A number of process names in 'processes' array and the number of identifiers in 'pids' array.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if at least one process was terminated, FALSE otherwise.
 The function reports the results to the standard console output.
*/

int com_proc_termination_report(char (*processes)[MAX_PATH],ULONG *pids,int cnt,PCOM_ERROR err_inf)
{
  int res=FALSE;

  int contains_valid_pid=FALSE;
  for (int i=0;i<cnt;i++) contains_valid_pid|=pids[i]!=COM_CID_INVALID;
  if (!contains_valid_pid) return res;

  printf("\nChecking termination results:\n");

  PROCESSENTRY32 pe;
  pe.dwSize=sizeof(pe);

  for (int w=0;(w<5) && !res;w++)
  {
    printf("\nWaiting for termination of the process(es) ...\n\n");
    Sleep(1000);

    HANDLE shot=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
    if (shot)
    {
      for (int i=0;i<cnt;i++)
      {
        if (pids[i]!=COM_CID_INVALID)
        {
          int found=FALSE;
          if (Process32First(shot,&pe))
          {
            do
            {
              if ((pe.th32ProcessID==pids[i]) && !stricmp(processes[i],pe.szExeFile))
              {
                found=TRUE;
                break;
              }
            } while (Process32Next(shot,&pe));

            if (!found)
            {
              res=TRUE;
              printf("Process \"%s\" (PID %ld) was successfully terminated.\n",processes[i],pids[i]);
            } else printf("Process \"%s\" (PID %ld) still exists.\n",processes[i],pids[i]);
          } else
          {
            com_err_set(err_inf,"Unable to get the first process from the snapshot.\n");
            break;
          }
        }
      }

      CloseHandle(shot);
    } else com_err_set(err_inf,"Unable to create toolhelp process snapshot.\n");
  }

  return res;
}





/*
 Prints downloaded data in case they contains magic pattern.

 'data' A pointer to a null terminated string that contains downloaded data.
 'size' A length of 'data' buffer.
 'conf' A pointer to a structure with the loaded configuration.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if 'data' contains conf->magic, FALSE otherwise.
*/

int com_find_pattern_and_print_data(char *data,size_t size,PCOM_CONF conf,PCOM_ERROR err_inf)
{
  int res=FALSE;
  data[size-1]='\0';

  int totsize=strlen(data);
  if (totsize)
  {
    fprintf(conout,"Here are the downloaded data:\n\n%s\n\n",data);
    res=strstr(data,conf->magic)!=NULL;
    if (!res) com_err_set_nc(err_inf,"Magic pattern not found.\n");
  } else com_err_set(err_inf,"No data has been received.\n");

  return res;
}


/*
 Adds access rights to the object for a specific user.

 'object' A handle to the object.
 'type' A type of the object (see 'ObjectType' argument of SetSecurityInfo API).
 'info' Security information (see 'SecurityInfo' argument of SetSecurityInfo API).
 'access' An access mask (see 'grfAccessPermissions' item of EXPLICIT_ACCESS structure).
 'acmode' An access mode (see 'grfAccessMode' item of EXPLICIT_ACCESS structure).
 'user' Identifies the user for which the access rights are set. If 'user' is NULL, the current user's name is used.
 'merge' If 'merge' is TRUE, merge the object's access rights with the access rights for the selected user.
 If 'merge' is FALSE, remove all access rights and only assign the access for the selected user.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_security_info_set(HANDLE object,SE_OBJECT_TYPE type,SECURITY_INFORMATION info,DWORD access,ACCESS_MODE acmode,char *user,int merge,PCOM_ERROR err_inf)
{
  int res=FALSE;

  PSECURITY_DESCRIPTOR sd=NULL;
  PACL acl=NULL;

  DWORD ret=ERROR_SUCCESS;

  if (merge)
    ret=GetSecurityInfo(object,type,info,NULL,NULL,&acl,NULL,&sd);

  if (ret==ERROR_SUCCESS)
  {
    EXPLICIT_ACCESS ea;
    char *name=user;

    char curuser[UNLEN+1];
    DWORD userlen=sizeof(curuser);

    ret=TRUE;
    if (!name)
    {
      name=curuser;
      ret=GetUserName(name,&userlen);
    }

    if (ret)
    {
      ea.grfAccessPermissions=access;
      ea.grfAccessMode=acmode;
      ea.grfInheritance=NO_INHERITANCE;
      ea.Trustee.pMultipleTrustee=NULL;
      ea.Trustee.MultipleTrusteeOperation=NO_MULTIPLE_TRUSTEE;
      ea.Trustee.TrusteeForm=TRUSTEE_IS_NAME;
      ea.Trustee.TrusteeType=TRUSTEE_IS_USER;
      ea.Trustee.ptstrName=name;

      PACL newacl=NULL;
      ret=SetEntriesInAcl(1,&ea,acl,&newacl);
      if (ret==ERROR_SUCCESS)
      {
        ret=SetSecurityInfo(object,type,info,NULL,NULL,newacl,NULL);

        if (ret==ERROR_SUCCESS) res=TRUE;
        else com_err_set_sc(err_inf,ret,"Unable to set a security descriptor to the object (handle 0x%p).\n",object);

        LocalFree(newacl);
      } else com_err_set_sc(err_inf,ret,"Unable to create new access control.\n");
    } else com_err_set(err_inf,"Unable to retrieve a name of the current user.\n");
  } else com_err_set_sc(err_inf,ret,"Unable to retrieve a security descriptor of the object (handle 0x%p).\n",object);

  return res;
}


/*
 Enables a privilege for current process' token.

 'priv_name' A name of the privilege.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_privilege_enable(char *priv_name,PCOM_ERROR err_inf)
{
  int res=FALSE;
  HANDLE tok;
  LUID luid;
  TOKEN_PRIVILEGES privs;

  if (OpenProcessToken(GetCurrentProcess(),TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY,&tok))
  {
    if (LookupPrivilegeValue(NULL,priv_name,&luid))
    {
      privs.PrivilegeCount=1;
      privs.Privileges[0].Luid=luid;
      privs.Privileges[0].Attributes=SE_PRIVILEGE_ENABLED;
      DWORD ret_len;
      res=AdjustTokenPrivileges(tok,0,&privs,sizeof(TOKEN_PRIVILEGES),NULL,&ret_len);
      if (!res)
        com_err_set(err_inf,"Unable to add \"%s\" to current process' token.\n",priv_name);
    } else com_err_set(err_inf,"Unable to lookup value of \"%s\" privilege.\n",priv_name);

    CloseHandle(tok);
  } else com_err_set(err_inf,"Unable to open current process' token.\n");

  return res;
}


/*
 Enables the debug privilege for current process' token.

 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_privilege_enable_debug(PCOM_ERROR err_inf)
{
  return com_privilege_enable(SE_DEBUG_NAME,err_inf);
}


/*
 Enables the shutdown privilege for current process' token.

 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_privilege_enable_shutdown(PCOM_ERROR err_inf)
{
  return com_privilege_enable(SE_SHUTDOWN_NAME,err_inf);
}


/*
 Enables the backup privilege for current process' token.

 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
*/

int com_privilege_enable_backup(PCOM_ERROR err_inf)
{
  return com_privilege_enable(SE_BACKUP_NAME,err_inf);
}


/*
 Enables the restore privilege for current process' token.

 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
*/

int com_privilege_enable_restore(PCOM_ERROR err_inf)
{
  return com_privilege_enable(SE_RESTORE_NAME,err_inf);
}


/*
 Copies contents of the source registry key to the destination key.

 'src_root' A handle to an open registry key to which 'src_subkey' is relative.
 'src_root' A pointer to a null-terminated string of the source subkey name.
 'dst_root' A handle to an open registry key to which 'dst_subkey' is relative.
 'src_root' A pointer to a null-terminated string of the destination subkey name.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_reg_key_copy(HKEY src_root,char *src_subkey,HKEY dst_root,char *dst_subkey)
{
  int res=FALSE;

  int error=FALSE;
  HKEY src;
  LONG ret=RegOpenKeyEx(src_root,src_subkey,0,KEY_READ,&src);
  if (ret==ERROR_SUCCESS)
  {
    HKEY dst;
    ret=RegCreateKeyEx(dst_root,dst_subkey,0,NULL,REG_OPTION_NON_VOLATILE,KEY_ALL_ACCESS,NULL,&dst,NULL);
    if (ret==ERROR_SUCCESS)
    {
      for (int i=0;;i++)
      {
        char vname[MAX_PATH],*vval=NULL;
        DWORD size_name=sizeof(vname),size_val=0,type;

        ret=RegEnumValue(src,i,vname,&size_name,NULL,&type,NULL,&size_val);
        if ((ret==ERROR_MORE_DATA) || (ret==ERROR_SUCCESS))
        {
          if (size_val) vval=malloc(size_val);
          if (vval || !size_val)
          {
            memset(vname,0,sizeof(vname));
            size_name=sizeof(vname);
            ret=RegEnumValue(src,i,vname,&size_name,NULL,&type,vval,&size_val);
            if (ret==ERROR_SUCCESS)
            {
              ret=RegSetValueEx(dst,vname,0,type,vval,size_val);
              error=ret!=ERROR_SUCCESS;
              if (error) break;
            } else
            {
              error=TRUE;
              break;
            }
          }
        } else
        {
          error=ret!=ERROR_NO_MORE_ITEMS;
          break;
        }
      }

      if (!error)
      {
        for (int i=0;;i++)
        {
          char kname[MAX_PATH],kclass[MAX_PATH];
          DWORD size=sizeof(kname),size_class=sizeof(kclass);
          memset(kname,0,sizeof(kname));
          memset(kclass,0,sizeof(kclass));
          FILETIME lwt;

          ret=RegEnumKeyEx(src,i,kname,&size,NULL,NULL,&size_class,&lwt);
          if (ret==ERROR_SUCCESS)
          {
            error=!com_reg_key_copy(src,kname,dst,kname);
            if (error) break;
          } else
          {
            error=ret!=ERROR_NO_MORE_ITEMS;
            break;
          }
        }
      }

      res=!error;
      RegCloseKey(dst);
    }

    RegCloseKey(src);
  }

  return res;
}


/*
 Converts name of a root key to its HKEY value.

 'root_name' One of these strings: "HKCR", "HKCU", "HKLM".
 'hkey' A variable that is filled with one of the following values:
   HKEY_CLASSES_ROOT    - if 'root_name' is "HKCR"
   HKEY_CURRENT_USER    - if 'root_name' is "HKCU"
   HKEY_LOCAL_MACHINE   - if 'root_name' is "HKLM"
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
 The function succeeds always if 'root_name' is one of the mentioned values.
*/

int com_reg_root_name_to_hkey(char *root_name,HKEY *hkey,PCOM_ERROR err_inf)
{
  *hkey=!strcmp(root_name,"HKCR") ? HKEY_CLASSES_ROOT :
        !strcmp(root_name,"HKCU") ? HKEY_CURRENT_USER :
        !strcmp(root_name,"HKLM") ? HKEY_LOCAL_MACHINE :
        NULL;

  if (!*hkey) com_err_set_nc(err_inf,"Unknown root key name \"%s\".\n",root_name);

  return *hkey!=NULL;
}


/*
 Disables a system service.

 'svc_name' A name of the service to disable.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_service_disable(char *svc_name,PCOM_ERROR err_inf)
{
  int res=FALSE;

  SC_HANDLE manager=OpenSCManager(NULL,NULL,SC_MANAGER_CONNECT);
  if (manager)
  {
    SC_HANDLE service=OpenService(manager,svc_name,SERVICE_CHANGE_CONFIG);
    if (service)
    {
      if (ChangeServiceConfig(service,SERVICE_NO_CHANGE,SERVICE_DISABLED,SERVICE_NO_CHANGE,NULL,NULL,NULL,NULL,NULL,NULL,NULL))
      {
        res=TRUE;
      } else com_err_set(err_inf,"Unable to change settings of \"%s\" service.\n",svc_name);

      CloseServiceHandle(service);
    } else com_err_set(err_inf,"Unable to open \"%s\" service.\n",svc_name);

    CloseServiceHandle(manager);
  } else com_err_set(err_inf,"Unable to connect service manager.\n");

  return res;
}



/*
 Obtains system information of a specific class.

 'iclass' The system information class, see 'SystemInformationClass' parameter of ZwQuerySystemInformation API.
 'table' A pointer to a variable that receives a pointer to a newly allocated buffer that contains
 the requested information.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
 If the function succeeds, 'table' points to the memory that must be freed using free() when no longer needed.
*/

int com_get_system_info_table(DWORD iclass,void **table,PCOM_ERROR err_inf)
{
  int res=FALSE;

  size_t buf_size=0x1000;       // initial size of allocated memory for the requested information
  for (;;)
  {
    void* buf=malloc(buf_size);
    if (!buf)
    {
      com_err_set(err_inf,"Unable to allocate %d bytes of memory.\n",buf_size);
      break;
    }

    NTSTATUS status=ZwQuerySystemInformation(iclass,buf,buf_size,NULL);

    if (NT_SUCCESS(status))
    {
      res=TRUE;
      *table=buf;
      break;
    }

    free(buf);
    buf_size=2*buf_size;

    if (status!=STATUS_INFO_LENGTH_MISMATCH)
    {
      com_err_set_sc(err_inf,RtlNtStatusToDosError(status),"Unable to query system information class %d.\n",buf_size);
      break;
    }
  }

  return res;
}
