/*

 Security Software Testing Suite - SSDT support shared code library
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/


 Credits:

   * The code of this library is based on code of SDT Restore by Tan Chew Keong,
     http://www.security.org.sg/code/sdtrestore.html.

   * The code of com_ntrknl_get_KiServiceTable_addr() based on the method of finding KiServiceTable by 90210,
     http://www.rootkit.com/newsread.php?newsid=176.

*/

#include <windows.h>
#include <stdio.h>
#include <imagehlp.h>
#include <aclapi.h>
#include <ddk/ntapi.h>
#include <ddk/ntifs.h>
#include "common.h"
#include "common-ssdt.h"

int com_ssdt_restore_from_user_mode(PCOM_ERROR err_inf);

int com_ntkrnl_get_active_kernel_info(char *krnl,void **base,PCOM_ERROR err_inf);
ULONG com_ntrknl_get_KiServiceTable_addr(void* addr,PIMAGE_NT_HEADERS hdr,ULONG ssdtaddr);
int com_ntkrnl_get_original_ssdt(void** table,int *params,int *count,ULONG *offset,PCOM_ERROR err_inf);
int com_ntdll_get_ntindex_by_addr(void* addr);
int com_ntdll_get_ssdt_index_max(int *max_index,PCOM_ERROR err_inf);
ULONG com_offset_to_rva_image(PLOADED_IMAGE image,ULONG base,ULONG offset);
void* com_get_proc_address_image(PLOADED_IMAGE image,char *sym);


/*
 Restores kernel mode SSDT hooks from the user mode using the system's physical memory section object.

 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_ssdt_restore_from_user_mode(PCOM_ERROR err_inf)
{
  int res=FALSE;

  UNICODE_STRING section_name;
  RtlInitUnicodeString(&section_name,COM_SSDT_PHYMEM_SECTION_NAME);

  OBJECT_ATTRIBUTES oa;
  InitializeObjectAttributes(&oa,&section_name,OBJ_CASE_INSENSITIVE,0,NULL);

  HANDLE section=NULL;
  NTSTATUS status=ZwOpenSection(&section,READ_CONTROL | WRITE_DAC,&oa);
  if (NT_SUCCESS(status))
  {
    if (com_security_info_set(section,SE_KERNEL_OBJECT,DACL_SECURITY_INFORMATION,SECTION_MAP_WRITE,GRANT_ACCESS,NULL,TRUE,err_inf))
    {
      CloseHandle(section);
      section=NULL;

      if (com_verbosity_get()) fprintf(conout,"Access rights to \"%S\" section modified.\n",COM_SSDT_PHYMEM_SECTION_NAME);

      status=ZwOpenSection(&section,SECTION_MAP_READ | SECTION_MAP_WRITE,&oa);
      if (NT_SUCCESS(status))
      {
        if (com_verbosity_get()) fprintf(conout,"Section \"%S\" opened for writing.\n",COM_SSDT_PHYMEM_SECTION_NAME);

        int count=0;
        void *table[COM_SSDT_MAX_FUNC_COUNT];
        int params[COM_SSDT_MAX_FUNC_COUNT];
        ULONG ssdt_offset;

        if (com_ntkrnl_get_original_ssdt(table,params,&count,&ssdt_offset,err_inf))
        {
          if (com_verbosity_get()) fprintf(conout,"%d system services found.\n",count);

          char ntkrnl[MAX_PATH];
          PVOID ntkrnl_base;
          if (com_ntkrnl_get_active_kernel_info(ntkrnl,&ntkrnl_base,err_inf))
          {
            void *ssdt_phymem_offset=(void*)(((size_t)ntkrnl_base & ~0xC0000000) + ssdt_offset);
            LARGE_INTEGER offset={.QuadPart=(ULONG)ssdt_phymem_offset};
            PVOID base=NULL;
            DWORD size=0x2000;
            status=ZwMapViewOfSection(section,GetCurrentProcess(),&base,0,size,&offset,&size,ViewShare,0,PAGE_READWRITE);
            if (NT_SUCCESS(status))
            {
              if (com_verbosity_get()) fprintf(conout,"Section \"%S\" mapped to the current process.\n",COM_SSDT_PHYMEM_SECTION_NAME);

              int fixed=0;
              void **mem_table=(void**)((size_t)base+(size_t)ssdt_phymem_offset-offset.LowPart);

              for (int i=0;i<count;i++)
                if (table[i]!=mem_table[i])
                {

                  if (com_verbosity_get()) fprintf(conout,"SSDT[%d] has been hooked (original function = 0x%p, current function = 0x%p) and will be fixed.\n",
                                                   i,table[i],mem_table[i]);
                  mem_table[i]=table[i];
                  fixed++;
                }


              if (com_verbosity_get()) fprintf(conout,"%d hooked system services found and restored.\n",fixed);

              res=TRUE;

              ZwUnmapViewOfSection(GetCurrentProcess(),base);
            } else com_err_set_sc(err_inf,RtlNtStatusToDosError(status),"Unable to map section \"%S\" to memory.\n",COM_SSDT_PHYMEM_SECTION_NAME);
          }
        }
      } else com_err_set_sc(err_inf,RtlNtStatusToDosError(status),"Unable to open section \"%S\" for writing.\n",COM_SSDT_PHYMEM_SECTION_NAME);
    }

    if (section) CloseHandle(section);
  } else com_err_set_sc(err_inf,RtlNtStatusToDosError(status),"Unable to open section \"%S\" for access change.\n",COM_SSDT_PHYMEM_SECTION_NAME);

  return res;
}



/*
 Retrieves information about the active kernel module.

 'krnl' A pointer to a buffer of at least MAX_PATH chars to which a full path to the kernel will be stored.
 'base' A pointer to a variable that will receive a memory base of the active kernel.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_ntkrnl_get_active_kernel_info(char *krnl,void **base,PCOM_ERROR err_inf)
{
  int res=FALSE;

  krnl[0]='\0';
  *base=NULL;

  char sysdir[MAX_PATH];
  if (com_get_system_path(sysdir,sizeof(sysdir),err_inf))
  {
    SYSTEM_MODULE_INFORMATION info;
    memset(&info,0,sizeof(info));
    ULONG size=sizeof(info);
    NTSTATUS status=ZwQuerySystemInformation(SystemModuleInformation,&info,size,&size);
    if (status==STATUS_INFO_LENGTH_MISMATCH) status=STATUS_SUCCESS;

    if (NT_SUCCESS(status))
    {
      res=TRUE;
      snprintf(krnl,MAX_PATH,"%s\\%s",sysdir,&info.Module[0].ImageName[info.Module[0].PathLength]);
      krnl[MAX_PATH-1]='\0';
      *base=info.Module[0].Base;
    } else com_err_set_sc(err_inf,RtlNtStatusToDosError(status),"Unable to get information about the kernel module.\n");
  }

  return res;
}


/*
 Finds KiServiceTable address in the NT kernel image.

 'addr' A pointer to the kernel image.
 'hdr' A pointer to NT image headers inside the kernel image.
 'ssdtaddr' Relative virtual address of Service Table in the kernel image.

 If the function succeeds the return value is address of KiServiceTable, otherwise it is 0.
*/

ULONG com_ntrknl_get_KiServiceTable_addr(void* addr,PIMAGE_NT_HEADERS hdr,ULONG ssdtaddr)
{
  ULONG res=0;

  DWORD ssdtbase=hdr->OptionalHeader.ImageBase+ssdtaddr;
  PIMAGE_SECTION_HEADER sec=IMAGE_FIRST_SECTION(hdr);

  for (int i=0;i<hdr->FileHeader.NumberOfSections && !res;i++,sec++)
  if (sec->Characteristics & (IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_CNT_CODE))
  {
    char *p=ImageRvaToVa(hdr,addr,sec->VirtualAddress,NULL);

    if (sec->SizeOfRawData)
    {
      for (char *q=p;q<p+sec->SizeOfRawData;q++)
        if ((*(WORD*)q==0x05C7) && (*(PULONG)(q+2)==ssdtbase))
        {
          res=(*(PULONG)(q+6)-hdr->OptionalHeader.ImageBase);
          break;
        }
    }
  }

  return res;
}



/*
 Retrieves original table of system services and size of their parameters.

 'table' A pointer to an array of at least COM_SSDT_MAX_FUNC_COUNT pointers
 that will be filled with pointers to original system services.
 'params' A pointer to an array of at least COM_SSDT_MAX_FUNC_COUNT integers
 that will be filled with parameter sizes of system services.
 'count' A pointer to a variable that receives a number of system services.
 'offset' Optionally, a pointer to a variable that receives offset of SSDT inside the kernel image.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_ntkrnl_get_original_ssdt(void** table,int *params,int *count,ULONG *offset,PCOM_ERROR err_inf)
{
  int idx_max;
  int res=FALSE;
  if (com_ntdll_get_ssdt_index_max(&idx_max,err_inf))
  {
    char ntkrnl[MAX_PATH],*used_image=ntkrnl;
    PVOID ntkrnl_base;
    if (com_ntkrnl_get_active_kernel_info(ntkrnl,&ntkrnl_base,err_inf))
    {
      if (com_verbosity_get()) fprintf(conout,"Active kernel base is 0x%p, active kernel path is \"%s\".\n",ntkrnl_base,ntkrnl);

      PLOADED_IMAGE image=ImageLoad(ntkrnl,NULL);
      if (!image)
      {
        DWORD error=GetLastError();

        char *sl=strrchr(ntkrnl,'\\');
        if (sl) sl++;
        else sl=ntkrnl;

        image=ImageLoad(sl,NULL);
        if (image)
        {
          used_image=sl;
          if (com_verbosity_get()) fprintf(conout,"A copy of \"%s\" used instead of \"%s\".\n",sl,ntkrnl);
        } else SetLastError(error);
      }

      if (image)
      {
        ULONG ssdtrva=(ULONG)com_get_proc_address_image(image,"KeServiceDescriptorTable")-(ULONG)image->MappedAddress;

        ULONG kist=com_ntrknl_get_KiServiceTable_addr(image->MappedAddress,image->FileHeader,ssdtrva);
        ULONG kipa=com_ntrknl_get_KiServiceTable_addr(image->MappedAddress,image->FileHeader,ssdtrva+COM_SSDT_PARAMS_OFFSET);

        if (kist)
        {
          if (com_verbosity_get()) fprintf(conout,"SSDT found at offset 0x%p.\n",(void*)kist);
          if (offset) *offset=kist;

          if (kipa)
          {
            PULONG functions=ImageRvaToVa(image->FileHeader,image->MappedAddress,kist,NULL);
            PUCHAR params_uchar=ImageRvaToVa(image->FileHeader,image->MappedAddress,kipa,NULL);

            for (int i=0;i<=idx_max;i++)
            {
              table[i]=(void*)(functions[i]-image->FileHeader->OptionalHeader.ImageBase+(ULONG)ntkrnl_base);
              params[i]=params_uchar[i];
            }

            *count=idx_max+1;
            res=TRUE;
          } else com_err_set_nc(err_inf,"Unable to find SSDT parameters array in the kernel image.\n");
        } else com_err_set_nc(err_inf,"Unable to find SSDT in the kernel image.\n");

        ImageUnload(image);
      } else if (used_image==ntkrnl) com_err_set(err_inf,"Unable to load kernel image \"%s\" to the memory, try to copy it to this test's directory and try the test again.\n",ntkrnl);
      else com_err_set(err_inf,"Unable to load kernel image \"%s\" to the memory.\n",ntkrnl);
    }
  }

  return res;
}


/*
 Extracts system service index from the user mode service stub.

 'addr' A pointer to the service stub.

 If the function succeeds the return value is the index of the system service.
 If the function fails the return value is -1.
*/

int com_ntdll_get_ntindex_by_addr(void* addr)
{
  UCHAR *caddr=addr;
  int ret=-1;
  if (caddr[0]==0xB8)
  {
    ULONG pat=*(PULONG)(&caddr[5]);
    if ((pat==0x0424548D)                                               // OS version < Windows XP
     || (pat==0xFE0300BA))                                              // OS version >= Windows XP
     ret=*(int*)(&caddr[1]);

    if ((ret<0) || (ret>COM_SSDT_MAX_FUNC_COUNT)) ret=-1;
  }
  return ret;
}



/*
 Retrieves the highest system services index by analysing "ntdll.dll".

 'max_index' A pointer to a variable that receives the highest system service index.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_ntdll_get_ssdt_index_max(int *max_index,PCOM_ERROR err_inf)
{
  int res=FALSE;

  int index=-1;
  char sysdir[MAX_PATH];
  if (com_get_system_path(sysdir,sizeof(sysdir),err_inf))
  {
    char ntdll[MAX_PATH],*used_dll=ntdll;
    snprintf(ntdll,MAX_PATH,"%s\\ntdll.dll",sysdir);
    ntdll[MAX_PATH-1]='\0';

    PLOADED_IMAGE image=ImageLoad(ntdll,NULL);
    if (!image)
    {
      DWORD error=GetLastError();
      image=ImageLoad("ntdll.dll",NULL);
      if (image)
      {
        used_dll="ntdll.dll";
        if (com_verbosity_get()) fprintf(conout,"A copy of \"ntdll.dll\" used instead of \"%s\".\n",ntdll);
      } else SetLastError(error);
    }

    if (image)
    {
      PIMAGE_DATA_DIRECTORY exp=image->FileHeader->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_EXPORT;

      ULONG module=(ULONG)image->MappedAddress;
      PVOID pmod=image->MappedAddress;
      ULONG exp_size=exp->Size;
      if (exp_size)
      {
        ULONG exp_addr=(ULONG)ImageRvaToVa(image->FileHeader,NULL,exp->VirtualAddress,NULL);
        PIMAGE_EXPORT_DIRECTORY exports=(PIMAGE_EXPORT_DIRECTORY)(module+exp_addr);
        PULONG functions = (PULONG)(ImageRvaToVa(image->FileHeader,pmod,exports->AddressOfFunctions,NULL));
        PSHORT ordinals  = (PSHORT)(ImageRvaToVa(image->FileHeader,pmod,exports->AddressOfNameOrdinals,NULL));

        for (int i=0;i<(int)exports->NumberOfNames;i++)
        {
          ULONG ord=ordinals[i];
          ULONG rawaddr=(ULONG)(ImageRvaToVa(image->FileHeader,pmod,functions[ord],NULL));
          if (!rawaddr) rawaddr=functions[ord]+module;

          int ntindex=com_ntdll_get_ntindex_by_addr((void*)rawaddr);
          if (ntindex>index) index=ntindex;
        }

        if (index!=-1)
        {
          *max_index=index;
          res=TRUE;
        } else com_err_set_nc(err_inf,"No native functions found in \"%s\".\n",used_dll);
      } else com_err_set_nc(err_inf,"Export section of \"%s\" is empty.\n",used_dll);

      ImageUnload(image);
    } else if (used_dll==ntdll) com_err_set(err_inf,"Unable to load \"%s\" to the memory, try to copy it to this test's directory and try the test again.\n",ntdll);
    else com_err_set(err_inf,"Unable to load \"%s\" to the memory.\n",ntdll);
  }

  return res;
}


/*
 Converts a raw image data offset to a virtual address.

 'image' A pointer to the loaded image structure (see ImageLoad API).
 'base' The image base.
 'offset' The offset to convert.

 If the offset is a part of some section of the image the return value is a sum of the virtual address of
 this section and a relative section position of the data referenced by the given offset,
 otherwise the return value is a sum of the given image base and the offset.
*/

ULONG com_offset_to_rva_image(PLOADED_IMAGE image,ULONG base,ULONG offset)
{
  PIMAGE_NT_HEADERS hdr=image->FileHeader;
  PIMAGE_SECTION_HEADER sec=IMAGE_FIRST_SECTION(hdr);

  ULONG ret=offset;
  for (int i=0;i<hdr->FileHeader.NumberOfSections;i++,sec++)
  {
    if ((sec->PointerToRawData<=offset) && (offset<=sec->PointerToRawData+sec->SizeOfRawData))
    {
      ULONG delta=offset-sec->PointerToRawData;
      ret=sec->VirtualAddress+delta;
      break;
    }
  }
  return ret;
}


/*
 Returns a pointer to the exported symbol of the loaded image.

 'image' A pointer to the loaded image structure (see ImageLoad API).
 'sym' A pointer to a null terminated name of the symbol.

 If the function succeeds the return value is a pointer to the exported symbol,
 otherwise the return value is NULL.
*/

void* com_get_proc_address_image(PLOADED_IMAGE image,char *sym)
{
  void* res=NULL;

  PIMAGE_DATA_DIRECTORY exp=image->FileHeader->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_EXPORT;
  if (exp)
  {
    ULONG module=(ULONG)image->MappedAddress;
    PVOID pmod=image->MappedAddress;
    ULONG exp_size=exp->Size;
    if (exp_size)
    {
      ULONG exp_addr=(ULONG)ImageRvaToVa(image->FileHeader,NULL,exp->VirtualAddress,NULL);
      PIMAGE_EXPORT_DIRECTORY exports=(PIMAGE_EXPORT_DIRECTORY)(module+exp_addr);
      if (exports && exports->AddressOfNames)
      {
        PULONG functions = (PULONG)(ImageRvaToVa(image->FileHeader,pmod,exports->AddressOfFunctions,NULL));
        PSHORT ordinals  = (PSHORT)(ImageRvaToVa(image->FileHeader,pmod,exports->AddressOfNameOrdinals,NULL));
        PULONG names     = (PULONG)(ImageRvaToVa(image->FileHeader,pmod,exports->AddressOfNames,NULL));

        for (int i=0;i<(int)exports->NumberOfNames;i++)
        {
          ULONG ord=ordinals[i];
          ULONG rawaddr=(ULONG)(ImageRvaToVa(image->FileHeader,pmod,functions[ord],NULL));
          char *fname=ImageRvaToVa(image->FileHeader,pmod,names[i],NULL);

          if (((functions[ord]<exp->VirtualAddress) || (functions[ord]>=exp->VirtualAddress+exp_size))
           && (!strcmp(fname,sym)))
          {
            ULONG offset=rawaddr ? rawaddr-(ULONG)pmod : functions[ord];
            ULONG uret=rawaddr ? com_offset_to_rva_image(image,(ULONG)pmod,offset) : offset;
            res=(PVOID)(uret+(ULONG)pmod);
            break;
          }
        }
      }
    }
  }

  return res;
}
