/*

 Security Software Testing Suite - Shared code library for leak-tests
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/


 Credits:

   * The code of com_attempt_dde() based on the code of Comodo Parent Injection Leak Test Suite - Test 3
     by Comodo Group, http://www.comodo.com/.

   * The code of com_attempt_check_ie_cache() based on the code of Surfer leak-test by Jarkko Turkulainen,
     http://www.klake.org/.

*/

#include <windows.h>
#include <stdio.h>
#include <winsock.h>
#include <wininet.h>
#include <shlguid.h>
#include <objbase.h>
#include <ddk/ntapi.h>
#include "ntinternals.h"
#include "common.h"
#include "dns.h"
#include "common-leak.h"


int com_get_default_browser_path(char *buffer,size_t size,PCOM_ERROR err_inf);
int com_get_default_browser_name(char *buffer,size_t size,PCOM_ERROR err_inf);
int com_get_default_browser_pid_wait(ULONG *pid,PCOM_ERROR err_inf);

int com_get_ie_path(char *buffer,size_t size,PCOM_ERROR err_inf);
int com_get_ie_pid_wait(ULONG *pid,PCOM_ERROR err_inf);

int com_attempt_wininet(char *buffer,size_t size,PCOM_CONF conf,PCOM_ERROR err_inf);
int com_attempt_dde(PCOM_CONF conf,int timeout,PCOM_ERROR err_inf);

int com_attempt_check_ie_cache(char *buffer,size_t size,PCOM_CONF conf,int timeout,PCOM_ERROR err_inf);

void com_uri_replace_eq(PCOM_CONF conf);



/*
 Obtains a path of the default browser, which is assumed to be a network allowed application.

 'buffer' A pointer to a caller allocated memory, to which the browser's path will be stored.
 'size' A size of 'buffer' in bytes.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_default_browser_path(char *buffer,size_t size,PCOM_ERROR err_inf)
{
  HKEY key;
  int res=FALSE;
  int ret=RegOpenKeyEx(HKEY_CLASSES_ROOT,"http\\shell\\open\\command",0,KEY_READ,&key);
  if (ret==ERROR_SUCCESS)
  {
    char name[MAX_PATH+40];
    DWORD lsize=sizeof(name);
    ret=RegQueryValueEx(key,"",NULL,NULL,name,&lsize);
    if (ret==ERROR_SUCCESS)
    {
      if (strlen(name))
      {
        char *dl=name;
        if (name[0]=='"')
        {
          do
          {
            dl[0]=dl[1];
            if (*dl!='"') dl++;
          } while (*dl && (*dl!='"'));
          if (*dl=='"') *dl='\0';
        } else
        {
          dl=strchr(name,' ');
          if (dl) *dl='\0';
        }

        res=strlen(name)>0;
        if (res)
        {
          res=GetLongPathName(name,buffer,size)!=0;
          if (!res) com_err_set(err_inf,"Unable to convert \"%s\" to the full path format.\n",name);
        } else com_err_set_nc(err_inf,"The default browser's path is empty.\n");
      } else com_err_set_nc(err_inf,"The default browser's path is invalid.\n");
    } else com_err_set_sc(err_inf,ret,"Unable to query the default value of \"HKCU\\http\\shell\\open\\command\".\n");

    RegCloseKey(key);
  } else com_err_set_sc(err_inf,ret,"Unable to open registry key \"HKCU\\http\\shell\\open\\command\".\n");

  return res;
}


/*
 Obtains a name of the default browser's main executable.

 'buffer' A pointer to a caller allocated memory, to which the browser's name will be stored.
 'size' A size of 'buffer' in bytes.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_default_browser_name(char *buffer,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  char buf[MAX_PATH];
  if (com_get_default_browser_path(buf,sizeof(buf),err_inf))
  {
    char *name=strrchr(buf,'\\');
    if (name) name++;
    else name=buf;

    lstrcpynA(buffer,name,size);
    res=TRUE;
  }

  return res;
}


/*
 Obtains process identifier of the default browser, which is assumed to be a network allowed application.
 If the browser is not running, the function asks user to run it and waits until it is started.

 'pid' A pointer to a variable that receives the default browser's process identifier.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_default_browser_pid_wait(ULONG *pid,PCOM_ERROR err_inf)
{
  int res=FALSE;
  char browser[MAX_PATH];
  if (com_get_default_browser_name(browser,sizeof(browser),err_inf))
  {
    if (com_get_pid_from_name(pid,browser,err_inf))
    {
      if (*pid==COM_CID_INVALID)
      {
        fprintf(conout,"The default browser (process name \"%s\") is not running, please run the default browser.\n",browser);

        for (;;)
        {
          if (!com_get_pid_from_name(pid,browser,err_inf)) break;

          Sleep(5000);

          if (*pid!=COM_CID_INVALID)
          {
            res=TRUE;
            break;
          }
        }
      } else res=TRUE;
    }
  }
  return res;
}


/*
 Obtains a path of Internet Explorer, which is assumed to be a network allowed application.

 'buffer' A pointer to a caller allocated memory, to which the Internet Explorer's path will be stored.
 'size' A size of 'buffer' in bytes.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_ie_path(char *buffer,size_t size,PCOM_ERROR err_inf)
{
  int res=FALSE;

  LPOLESTR iestr;
  if (StringFromCLSID(&CLSID_InternetExplorer,&iestr)==S_OK)
  {
    char subkey[MAX_PATH];
    snprintf(subkey,sizeof(subkey),"CLSID\\%S\\LocalServer32",iestr);
    subkey[sizeof(subkey)-1]='\0';

    HKEY hkey;
    LONG ret=RegOpenKeyEx(HKEY_CLASSES_ROOT,subkey,0,KEY_READ,&hkey);
    if (ret==ERROR_SUCCESS)
    {
      char data[MAX_PATH+2];
      memset(data,0,sizeof(data));
      DWORD data_size=sizeof(data);
      ret=RegQueryValueEx(hkey,NULL,NULL,NULL,data,&data_size);
      if (ret==ERROR_SUCCESS)
      {
        data[sizeof(data)-1]='\0';

        char *datacpy=data;
        if (datacpy[0]=='"')
        {
          datacpy++;
          char *quot=strrchr(datacpy,'"');
          if (quot) *quot='\0';
          data_size=strlen(datacpy);
        }

        if (data_size>0)
        {
          lstrcpynA(buffer,datacpy,size);
          buffer[size-1]='\0';

          res=TRUE;
        }
      } else com_err_set_sc(err_inf,ret,"Unable to query the default value of registry key \"HKCR\\%s\".\n",subkey);

      RegCloseKey(hkey);
    } else com_err_set_sc(err_inf,ret,"Unable to open registry key \"HKCR\\%s\".\n",subkey);

    CoTaskMemFree(iestr);
  } else com_err_set(err_inf,"Unable to convert Internet Explorer's CLSID to string.\n");

  return res;
}


/*
 Obtains process identifier of an instance of Internet Explorer (process "iexplore.exe").
 If Internet Explorer is not running, the function asks user to run it and waits until it is started.

 'pid' A pointer to a variable that receives the process identifier of Internet Explorer.
 'err_inf' A pointer to a structure that will be filled with an error message and a code if an error occurs.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_get_ie_pid_wait(ULONG *pid,PCOM_ERROR err_inf)
{
  int res=FALSE;

  if (com_get_pid_from_name(pid,"iexplore.exe",err_inf))
  {
    if (*pid==COM_CID_INVALID)
    {
      fprintf(conout,"Internet Explorer (process name \"iexplore.exe\") is not running, please run Internet Explorer.\n");

      for (;;)
      {
        if (!com_get_pid_from_name(pid,"iexplore.exe",err_inf)) break;

        Sleep(5000);

        if (*pid!=COM_CID_INVALID)
        {
          res=TRUE;
          break;
        }
      }
    } else res=TRUE;
  }

  return res;
}




/*
 Tries to establish Internet connection and download contents of the given web page
 using WinINet functions. If the function succeed the web page contents are returned to the caller.

 'buffer' A pointer to a caller allocated memory where the web page contents will be stored.
 'size' A size of 'buffer'.
 'conf' A pointer to the configuration structure filled by com_init_*().
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
*/

int com_attempt_wininet(char *buffer,size_t size,PCOM_CONF conf,PCOM_ERROR err_inf)
{
  int res=FALSE;

  HINTERNET inet=InternetOpen(COM_INET_AGENT,INTERNET_OPEN_TYPE_PRECONFIG,NULL,NULL,0);
  if (inet)
  {
    HINTERNET iurl=InternetOpenUrl(inet,conf->uri,NULL,0,INTERNET_FLAG_NO_CACHE_WRITE,0);
    if (iurl)
    {
      char *bufptr=buffer;
      int totsize=0;
      DWORD rsize=0;

      while (InternetReadFile(iurl,bufptr,size-totsize-1,&rsize) && (rsize>0))
      {
        bufptr+=rsize;
        totsize+=rsize;
      }

      *bufptr='\0';
      res=TRUE;

      InternetCloseHandle(iurl);
    } else com_err_set(err_inf,"Unable to open Internet page \"%s\" using InternetOpenUrl.\n",conf->uri);

    InternetCloseHandle(inet);
  } else com_err_set(err_inf,"Unable to open Internet connection using InternetOpen.\n");

  return res;
}



/*
 DDE callback function that does nothing.
 See 'pfnCallback' parameter of DdeInitialize API for more information.
*/

HDDEDATA CALLBACK DdeCallback(UINT uType,UINT uFmt,HCONV hconv,HDDEDATA hsz1,HDDEDATA hsz2,HDDEDATA hdata,HDDEDATA dwData1,HDDEDATA dwData2)
{
  return uType==XCLASS_BOOL ? (HDDEDATA)TRUE : (HDDEDATA)0;
}


/*
 Tries to redirect an existing instance of Internet Explorer to a given web page.
 The webpage contents can be obtained by com_attempt_check_ie_cache().

 'conf' A pointer to the configuration structure filled by com_init_*().
 'timeout' Time-out interval, in milliseconds.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
*/

int com_attempt_dde(PCOM_CONF conf,int timeout,PCOM_ERROR err_inf)
{
  int res=FALSE;

  DWORD id=0;
  UINT ret=DdeInitialize(&id,(PFNCALLBACK)DdeCallback,APPCLASS_STANDARD | CBF_FAIL_SELFCONNECTIONS,0);
  if (ret==DMLERR_NO_ERROR)
  {
    HSZ server=DdeCreateStringHandle(id,"Iexplore",CP_WINANSI);
    if (server)
    {
      HSZ topic=DdeCreateStringHandle(id,"WWW_OpenURL",CP_WINANSI);
      if (topic)
      {
        HSZ url=DdeCreateStringHandle(id,conf->uri,CP_WINANSI);
        if (url)
        {
          CONVCONTEXT ctx;
          memset(&ctx,0,sizeof(ctx));
          ctx.cb=sizeof(ctx);

          HCONV con=NULL;
          int ltimeout=timeout;
          while ((ltimeout>0) && !con)
          {
            ltimeout-=50;
            con=DdeConnect(id,server,topic,&ctx);
            if (con)
            {
              HDDEDATA ddedata=DdeClientTransaction(NULL,-1,con,url,CF_TEXT,XTYP_REQUEST,5000,NULL);
              if (ddedata)
              {
                res=TRUE;
              } else com_err_set_nc(err_inf,"DDE transaction failed, DDE error code %u.\n",DdeGetLastError(id));

              DdeDisconnect(con);
            } else if (ltimeout<=0) com_err_set_nc(err_inf,"Unable to connect \"Iexplore\" DDE server, DDE error code %u. Make sure Internet Explorer is running.\n",DdeGetLastError(id));
            else Sleep(50);
          }

          DdeFreeStringHandle(id,topic);
        } else com_err_set_nc(err_inf,"Unable to create DDE string handle for \"%s\", DDE error code %u.\n",conf->uri,DdeGetLastError(id));

        DdeFreeStringHandle(id,topic);
      } else com_err_set_nc(err_inf,"Unable to create DDE string handle for \"WWW_OpenUrl\", DDE error code %u.\n",DdeGetLastError(id));

      DdeFreeStringHandle(id,server);
    } else com_err_set_nc(err_inf,"Unable to create DDE string handle for \"Iexplore\", DDE error code %u.\n",DdeGetLastError(id));

    DdeUninitialize(id);
  } else com_err_set_nc(err_inf,"Unable to initialize DDE, DDE error code %u.\n",ret);

  return res;
}


/*
 Checks whether Internet Explorer reached its target URL specified in common configuration.
 If so, the function returns the downloaded data from the Internet Explorer's cache.

 'buffer' A pointer to a caller allocated memory where the web page contents will be stored.
 'size' A size of 'buffer'.
 'conf' A pointer to the configuration structure filled by com_init_*().
 'timeout' Time-out interval, in milliseconds.
 'err_inf' A pointer to a structure that will be filled with error message and code if an error occurs.

 The return value is TRUE if the function succeed, FALSE otherwise.
*/

int com_attempt_check_ie_cache(char *buffer,size_t size,PCOM_CONF conf,int timeout,PCOM_ERROR err_inf)
{
  int res=FALSE;
  DWORD cache_size=0;

  while ((timeout>0) && !cache_size)
  {
    if (!GetUrlCacheEntryInfo(conf->uri,NULL,&cache_size) && (GetLastError()==ERROR_INSUFFICIENT_BUFFER))
    {
      LPINTERNET_CACHE_ENTRY_INFO cache=malloc(cache_size);
      if (cache)
      {
        if (GetUrlCacheEntryInfo(conf->uri,cache,&cache_size))
        {
          HANDLE file=CreateFile(cache->lpszLocalFileName,GENERIC_READ,FILE_SHARE_READ | FILE_SHARE_WRITE,NULL,OPEN_EXISTING,0,NULL);
          if (file!=INVALID_HANDLE_VALUE)
          {
            DWORD bytes;
            memset(buffer,0,size);
            if (ReadFile(file,buffer,size,&bytes,NULL)) res=TRUE;
            else com_err_set(err_inf,"Unable to read data from file \"%s\".\n",cache->lpszLocalFileName);

            CloseHandle(file);
          } else com_err_set(err_inf,"Unable to open file \"%s\" for reading.\n",cache->lpszLocalFileName);
        } else com_err_set(err_inf,"Unable to get URL cache entry for \"%s\".\n",conf->url);

        free(cache);
      } else com_err_set_nc(err_inf,"Unable to allocate %ld bytes of memory.\n",cache_size);
    } else if ((GetLastError()==ERROR_FILE_NOT_FOUND) && (timeout>0))
    {
      timeout-=50;
      Sleep(50);
    } else
    {
      com_err_set(err_inf,"Unable to get size of URL cache entry for \"%s\".\n",conf->url);
      break;
    }
  }

  return res;
}


/*
 Replaces '=' characters in configuration's URI.

 This function is used in a few tests that does not support '=' in URI.

 'conf' A pointer to the configuration structure filled by com_init_*(), in which 'uri'
 member will be replaced not to contain '='.
*/

void com_uri_replace_eq(PCOM_CONF conf)
{
  char urieq[3*COM_INET_URI_MAX_LEN],*p=conf->uri,*q=urieq;
  while (*p && *p!=':') *q++=*p++;
  while (*p)
  {
    if (*p=='=')
    {
      *q++='%';
      *q++='3';
      *q='D';
    } else *q=*p;
    p++;
    q++;
  }
  *q='\0';

  lstrcpynA(conf->uri,urieq,sizeof(conf->uri));
  return;
}
