/*

 Security Software Testing Suite - Unhooking shared code library
 Copyright by www.matousec.com, Different Internet Experience Ltd.
 http://www.matousec.com/

*/

#include <stdio.h>
#include <windows.h>
#include "ntinternals.h"
#include "common.h"
#include "common-hook.h"


ULONG com_image_rva_to_offset_nthdr(PIMAGE_NT_HEADERS hdr,DWORD rva,int code_only);
void com_image_reloc_fixup(PIMAGE_NT_HEADERS hdr,PIMAGE_BASE_RELOCATION relocs,DWORD base_real,DWORD base_mem);
int com_hook_load_libraries(char** dll_names,int dll_count);



/*
 Converts a relative virtual address to a raw image data offset.

 'hdr' A pointer to the image NT headers.
 'rva' The relative virtual address to convert.
 'code_only' If set to TRUE, search for the address in code sections only.

 The return value is the raw data offset that corresponds to the given relative virtual address.
 If the function fails, including the case when 'code_only' is TRUE and 'rva' corresponds to an address
 outside code sections, the return value is 0.
*/

ULONG com_image_rva_to_offset_nthdr(PIMAGE_NT_HEADERS hdr,DWORD rva,int code_only)
{
  ULONG ret=0;
  PIMAGE_SECTION_HEADER sec=IMAGE_FIRST_SECTION(hdr);

  for (int i=0;i<hdr->FileHeader.NumberOfSections;i++,sec++)
  {
    if ((sec->VirtualAddress<=rva) && (sec->VirtualAddress+sec->SizeOfRawData>rva) && sec->SizeOfRawData)
    {
      ret=sec->PointerToRawData+rva-sec->VirtualAddress;
      if (code_only & !(sec->Characteristics & (IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_CNT_CODE))) ret=0;
      break;
    }
  }
  return ret;
}


/*
 Fixes relocations in the given image.

 'hdr' A pointer to the image NT headers.
 'relocs' A pointer to relocation table.
 'base_real' A real base of the image in memory.
 'base_mem' A relocation base.
*/

void com_image_reloc_fixup(PIMAGE_NT_HEADERS hdr,PIMAGE_BASE_RELOCATION relocs,DWORD base_real,DWORD base_mem)
{
  DWORD base_original=hdr->OptionalHeader.ImageBase;
  DWORD delta=base_mem-base_original;

  ULONG blocksize;
  while ((blocksize=relocs->SizeOfBlock))
  {
    ULONG pagerva=relocs->VirtualAddress;
    ULONG entries=(blocksize-sizeof(IMAGE_BASE_RELOCATION))/sizeof(WORD);

    PWORD re=(PWORD)((DWORD)relocs+sizeof(IMAGE_BASE_RELOCATION));
    while (entries>0)
    {
      int re_type=*re >> 12;
      WORD re_ofs=*re & 0x0FFF;

      DWORD offset=com_image_rva_to_offset_nthdr(hdr,re_ofs+pagerva,FALSE);
      if (offset)
      {
        offset+=base_real;

        switch (re_type)
        {
          case IMAGE_REL_BASED_HIGH:
            *((PWORD)offset)+=HIWORD(delta);
            break;

          case IMAGE_REL_BASED_LOW:
            *((PWORD)offset)+=LOWORD(delta);
            break;

          case IMAGE_REL_BASED_HIGHLOW:
            *((PDWORD)offset)+=delta;
            break;
        }
      }
      re++;
      entries--;
    }
    relocs=(PIMAGE_BASE_RELOCATION)((DWORD)relocs+blocksize);
  }

  return;
}


/*
 Loads DLLs and unhooks possible inline, IAT and EAT hooks.
 This function prints errors on standard error output.
 Does nothing if global 'unhook' variable is not set.

 'dll_names' An array of pointers to null-terminate names of DLLs to load. This array of DLLs
 must be sorted by dependencies. The i-th DLL in the array should not import any function
 from DLLs that appear on (i+1)-th to 'dll_count' positions.
 'dll_count' A number of items in 'names' array.

 If the function succeeds the return value is TRUE, otherwise it is FALSE.
*/

int com_hook_load_libraries(char** dll_names,int dll_count)
{
  if (!com_unhook_get()) return TRUE;

  int res=FALSE;

  int error=FALSE;
  int count=dll_count+1;

  COM_ERROR err_inf;
  err_inf.occurred=FALSE;

  HMODULE modules[count];
  char *names[count];

  for (int i=0;i<count;i++) modules[i]=NULL;
  memcpy(&names[0],dll_names,dll_count*sizeof(char*));

  char main_module[MAX_PATH];

  error=!com_get_module_name(NULL,main_module,sizeof(main_module),&err_inf);
  if (!error)
  {
    names[count-1]=main_module;
    modules[count-1]=GetModuleHandle(main_module);
    if (modules[count-1])
    {
      for (int i=0;i<count-1;i++)
      {
        modules[i]=LoadLibrary(names[i]);

        if (!modules[i])
        {
          com_err_set(&err_inf,"Unable to load library \"%s\".\n",names[i]);
          error=TRUE;
          break;
        }
      }
    } else
    {
      error=TRUE;
      com_err_set(&err_inf,"Unable to get handle of main module \"%s\".\n",main_module);
    }
  }

  if (!error)
  {
    char sysdir[MAX_PATH];
    if (com_get_system_path(sysdir,sizeof(sysdir),&err_inf))
    {
      for (int k=0;k<count;k++)
      {
        HMODULE module=modules[k];
        char *name=names[k];

        if (com_verbosity_get()) fprintf(conout,"Loading \"%s\" ...\n",name);

        char full_path[MAX_PATH],*used_path=full_path;
        if (k<count-1)
        {
          snprintf(full_path,MAX_PATH,"%s\\%s",sysdir,name);
          full_path[MAX_PATH-1]='\0';
        } else lstrcpynA(full_path,name,sizeof(full_path));

        HANDLE disk_file=CreateFile(full_path,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,0,NULL);
        if ((k<count-1) && (disk_file==INVALID_HANDLE_VALUE))
        {
          DWORD last_error=GetLastError();
          disk_file=CreateFile(name,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,0,NULL);;
          if (disk_file!=INVALID_HANDLE_VALUE)
          {
            used_path=name;
            if (com_verbosity_get()) fprintf(conout,"A copy of \"%s\" used instead of original file \"%s\".\n",name,full_path);
          } else SetLastError(last_error);
        }

        if (disk_file!=INVALID_HANDLE_VALUE)
        {
          DWORD disk_file_size=GetFileSize(disk_file,NULL);
          if (disk_file_size!=INVALID_FILE_SIZE)
          {
            void *disk_image=VirtualAlloc(NULL,disk_file_size,MEM_COMMIT,PAGE_READWRITE);
            if (disk_image)
            {
              DWORD bytes;
              if (ReadFile(disk_file,disk_image,disk_file_size,&bytes,NULL))
              {
                PIMAGE_DOS_HEADER dos_hdr=(PIMAGE_DOS_HEADER)disk_image;
                PIMAGE_NT_HEADERS nt_hdr=(PIMAGE_NT_HEADERS)((size_t)disk_image+dos_hdr->e_lfanew);
                PIMAGE_DATA_DIRECTORY exp=nt_hdr->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_EXPORT;
                PIMAGE_DATA_DIRECTORY rel=nt_hdr->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_BASERELOC;
                PIMAGE_DATA_DIRECTORY imp=nt_hdr->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_IMPORT;

                ULONG imp_size=imp->Size;
                ULONG exp_size=exp->Size;
                ULONG rel_size=rel->Size;

                ULONG rel_offs=com_image_rva_to_offset_nthdr(nt_hdr,rel->VirtualAddress,FALSE);
                ULONG rel_addr=rel_offs+(size_t)disk_image;

                ULONG exp_offs=com_image_rva_to_offset_nthdr(nt_hdr,exp->VirtualAddress,FALSE);
                ULONG exp_addr=exp_offs+(size_t)disk_image;

                ULONG imp_offs=com_image_rva_to_offset_nthdr(nt_hdr,imp->VirtualAddress,FALSE);
                ULONG imp_addr=imp_offs+(size_t)disk_image;


                if (rel_offs && rel_size) com_image_reloc_fixup(nt_hdr,(void*)rel_addr,(DWORD)disk_image,(DWORD)module);

                // find and fix inline and EAT hooks
                if (exp_offs && exp_size)
                {
                  PIMAGE_EXPORT_DIRECTORY exports=(PIMAGE_EXPORT_DIRECTORY)exp_addr;
                  PULONG functions = (PULONG)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,exports->AddressOfFunctions,FALSE));
                  PSHORT ordinals  = (PSHORT)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,exports->AddressOfNameOrdinals,FALSE));
                  PULONG names     = (PULONG)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,exports->AddressOfNames,FALSE));

                  PIMAGE_DOS_HEADER mem_dos_hdr=(PIMAGE_DOS_HEADER)module;
                  PIMAGE_NT_HEADERS mem_nt_hdr=(PIMAGE_NT_HEADERS)((size_t)module+mem_dos_hdr->e_lfanew);
                  PIMAGE_DATA_DIRECTORY mem_exp=mem_nt_hdr->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_EXPORT;

                  void *mem_exp_addr=(void*)((size_t)module+mem_exp->VirtualAddress);
                  PIMAGE_EXPORT_DIRECTORY mem_exports=(PIMAGE_EXPORT_DIRECTORY)mem_exp_addr;
                  PULONG mem_functions=(void*)((size_t)module+mem_exports->AddressOfFunctions);

                  for (int i=0;i<(int)exports->NumberOfNames;i++)
                  {
                    ULONG ord=ordinals[i];
                    if ((functions[ord]<exp->VirtualAddress) || (functions[ord]>=exp->VirtualAddress+exp_size))
                    {
                      DWORD rva=functions[ord];
                      DWORD raw=com_image_rva_to_offset_nthdr(nt_hdr,rva,TRUE);
                      void *rawaddr=(void*)((size_t)disk_image+raw);
                      void *memaddr=(void*)((size_t)module+rva);

                      char *fname=(char*)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,names[i],FALSE));

                      if (raw && ((size_t)raw<disk_file_size-COM_HOOK_COMPARE_BLOCK_SIZE) && memcmp(memaddr,rawaddr,COM_HOOK_COMPARE_BLOCK_SIZE))
                      {
                        UCHAR dumporg[COM_HOOK_COMPARE_BLOCK_SIZE*3];
                        UCHAR dumpmod[COM_HOOK_COMPARE_BLOCK_SIZE*3];

                        for (int j=0;j<COM_HOOK_COMPARE_BLOCK_SIZE;j++)
                        {
                          UCHAR p=*((UCHAR *)rawaddr+j);
                          UCHAR q=*((UCHAR *)memaddr+j);
                          snprintf(&dumporg[j*3],4,"%.2x ",p);
                          snprintf(&dumpmod[j*3],4,"%.2x ",q);
                        }

                        if (com_verbosity_get())
                          fprintf(conout,"  %s (0x%p) hooked!\n"
                                         "    Original bytes: %s\n"
                                         "    Modified bytes: %s\n",fname,memaddr,dumporg,dumpmod);

                        DWORD written;
                        if (!WriteProcessMemory(GetCurrentProcess(),memaddr,rawaddr,COM_HOOK_COMPARE_BLOCK_SIZE,&written))
                        {
                          com_err_set(&err_inf,"Unable to fix hooked %s function %s at 0x%p.\n",name,fname,memaddr);
                          break;
                        } else
                        {
                          if (memcmp(memaddr,rawaddr,COM_HOOK_COMPARE_BLOCK_SIZE))
                          {
                            fprintf(conout,"    Unable to fix bytes via WriteProcessMemory call, trying the hard way ...\n");

                            HANDLE proc=OpenProcess(PROCESS_ALL_ACCESS,FALSE,GetCurrentProcessId());
                            if (proc)
                            {
                              DWORD protect;
                              if (VirtualProtectEx(proc,memaddr,COM_HOOK_COMPARE_BLOCK_SIZE,PAGE_EXECUTE_READWRITE,&protect))
                              {
                                memcpy(memaddr,rawaddr,COM_HOOK_COMPARE_BLOCK_SIZE);
                                if (memcmp(memaddr,rawaddr,COM_HOOK_COMPARE_BLOCK_SIZE))
                                  fprintf(conout,"    Not even the hard way did the stuff.\n");

                                DWORD protect2;
                                VirtualProtectEx(proc,memaddr,COM_HOOK_COMPARE_BLOCK_SIZE,protect,&protect2);
                              } else
                              {
                                com_err_set(&err_inf,"Unable to change memory protection at 0x%p.\n",memaddr);
                                break;
                              }

                              CloseHandle(proc);
                            } else
                            {
                              com_err_set(&err_inf,"Unable to open own process.\n");
                              break;
                            }
                          }
                        }
                      } else
                      {
                        void *fgpa=GetProcAddress(module,fname);
                        void *cur_exp=(void*)((size_t)module+mem_functions[ord]);

                        if ((cur_exp==memaddr) && (fgpa!=cur_exp))
                        {
                          if (com_verbosity_get())
                            fprintf(conout,"  %s (0x%p) hooked!\n"
                                           "    Function redirected by GetProcAddress to 0x%p\n",fname,memaddr,fgpa);
                        } else if (cur_exp!=memaddr)
                        {
                          if (com_verbosity_get())
                            fprintf(conout,"  %s (0x%p) hooked!\n"
                                           "    Function redirected in EAT to 0x%p\n",fname,memaddr,fgpa);

                          DWORD written;
                          DWORD new_val=(ULONG)memaddr-(size_t)module;
                          if (!WriteProcessMemory(GetCurrentProcess(),&mem_functions[ord],&new_val,sizeof(new_val),&written))
                          {
                            com_err_set(&err_inf,"Unable to fix EAT hook at 0x%p, %s function %s at 0x%p.\n",
                                        &mem_functions[ord],name,fname,memaddr);
                            break;
                          }
                        }
                      }
                    }
                  }
                }

                // find and fix IAT hooks of other DLLs
                if (imp_offs && imp_size)
                {
                  PIMAGE_IMPORT_DESCRIPTOR imports=(PIMAGE_IMPORT_DESCRIPTOR)imp_addr;

                  while (imports->Name)
                  {
                    char *imp_name=(char *)(size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,imports->Name,FALSE);

                    HMODULE imp_mod=NULL;
                    for (int i=0;i<count;i++)
                      if ((i!=k) && !stricmp(imp_name,names[i]))
                      {
                        if (com_verbosity_get() && (i>k))
                          fprintf(conout,"  WARNING: \"%s\" imports from \"%s\", which has not been unhooked yet!\n",name,imp_name);

                        imp_mod=modules[i];
                        break;
                      }

                    if (imp_mod)
                    {
                      PIMAGE_DOS_HEADER imp_dos_hdr=(PIMAGE_DOS_HEADER)imp_mod;
                      PIMAGE_NT_HEADERS imp_nt_hdr=(PIMAGE_NT_HEADERS)((size_t)imp_mod+imp_dos_hdr->e_lfanew);
                      void *imp_start=imp_mod;
                      void *imp_end=(void *)((size_t)imp_mod+imp_nt_hdr->OptionalHeader.SizeOfImage);

                      PIMAGE_DATA_DIRECTORY imp_exp=imp_nt_hdr->OptionalHeader.DataDirectory+IMAGE_DIRECTORY_ENTRY_EXPORT;
                      void *imp_exp_addr=(void*)((size_t)imp_mod+imp_exp->VirtualAddress);
                      ULONG imp_exp_size=imp_exp->Size;

                      PIMAGE_EXPORT_DIRECTORY imp_exports=(PIMAGE_EXPORT_DIRECTORY)imp_exp_addr;
                      PULONG imp_functions = (void*)((size_t)imp_mod+imp_exports->AddressOfFunctions);
                      PSHORT imp_ordinals  = (void*)((size_t)imp_mod+imp_exports->AddressOfNameOrdinals);
                      PULONG imp_names     = (void*)((size_t)imp_mod+imp_exports->AddressOfNames);

                      if (imp_exp_addr)
                      {
                        PIMAGE_THUNK_DATA disk_import_list=(PIMAGE_THUNK_DATA)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,imports->OriginalFirstThunk,FALSE));
                        PIMAGE_THUNK_DATA disk_thunk_list=(PIMAGE_THUNK_DATA)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,imports->FirstThunk,FALSE));
                        PIMAGE_THUNK_DATA mem_import_list=(PIMAGE_THUNK_DATA)((size_t)module+imports->OriginalFirstThunk);
                        PIMAGE_THUNK_DATA mem_thunk_list=(PIMAGE_THUNK_DATA)((size_t)module+imports->FirstThunk);

                        while (disk_import_list->u1.Ordinal)
                        {
                          // import by ordinal
                          if (IMAGE_SNAP_BY_ORDINAL(disk_import_list->u1.Ordinal))
                          {
                            int ordinal=IMAGE_ORDINAL(disk_import_list->u1.Ordinal);

                            void *org_faddr=(void*)((size_t)imp_mod+imp_functions[ordinal]);
                            if ((imp_functions[ordinal]<imp_exp->VirtualAddress) || (imp_functions[ordinal]>=imp_exp->VirtualAddress+imp_exp_size))
                            {
                              void *mem_faddr=(void *)mem_thunk_list->u1.Function;

                              if ((org_faddr!=mem_faddr) && ((mem_faddr<imp_start) || (mem_faddr>imp_end)))
                              {
                                if (com_verbosity_get())
                                  fprintf(conout,"  Ordinal import %s:%d (0x%p) hooked in \"%s\"!\n"
                                                 "    Function redirected in IAT to 0x%p\n",imp_name,ordinal,org_faddr,name,mem_faddr);

                                DWORD written;
                                DWORD new_val=(ULONG)org_faddr;
                                if (!WriteProcessMemory(GetCurrentProcess(),&mem_thunk_list->u1.Function,&new_val,sizeof(new_val),&written))
                                {
                                  com_err_set(&err_inf,"Unable to fix ordinal IAT hook at 0x%p, DLL %s, function %d at 0x%p.\n",
                                              &mem_thunk_list->u1.Function,imp_name,ordinal,mem_faddr);
                                  break;
                                }
                              }
                            }
                          } else
                          {
                            // import by name
                            PIMAGE_IMPORT_BY_NAME nameimp=(PIMAGE_IMPORT_BY_NAME)((size_t)disk_image+com_image_rva_to_offset_nthdr(nt_hdr,disk_import_list->u1.AddressOfData,FALSE));
                            char *proc_name=nameimp->Name;

                            int ord=-1;
                            for (int i=0;i<(int)imp_exports->NumberOfNames;i++)
                            {
                              char *fname=(void*)((size_t)imp_mod+imp_names[i]);
                              if (fname && !stricmp(fname,proc_name))
                              {
                                ord=imp_ordinals[i];
                                break;
                              }
                            }

                            if (ord!=-1)
                            {
                              void *org_faddr=(void*)((size_t)imp_mod+imp_functions[ord]);
                              if ((imp_functions[ord]<imp_exp->VirtualAddress) || (imp_functions[ord]>=imp_exp->VirtualAddress+imp_exp_size))
                              {
                                void *mem_faddr=(void *)mem_thunk_list->u1.Function;

                                if ((org_faddr!=mem_faddr) && ((mem_faddr<imp_start) || (mem_faddr>imp_end)))
                                {
                                  if (com_verbosity_get())
                                    fprintf(conout,"  Name import %s:%s (0x%p) hooked in \"%s\"!\n"
                                                   "    Function redirected in IAT to 0x%p\n",imp_name,proc_name,org_faddr,name,mem_faddr);
                                }

                                DWORD written;
                                DWORD new_val=(ULONG)org_faddr;
                                if (!WriteProcessMemory(GetCurrentProcess(),&mem_thunk_list->u1.Function,&new_val,sizeof(new_val),&written))
                                {
                                  com_err_set(&err_inf,"Unable to fix IAT hook at 0x%p, DLL %s, function %s at 0x%p.\n",
                                              &mem_thunk_list->u1.Function,imp_name,proc_name,mem_faddr);
                                  break;
                                }
                              }
                            }
                          }

                          disk_import_list++;
                          disk_thunk_list++;
                          mem_import_list++;
                          mem_thunk_list++;
                        }
                      }
                    }

                    imports++;
                  }
                }

              } else com_err_set(&err_inf,"Unable to read file \"%s\".\n",used_path);

              VirtualFree(disk_image,0,MEM_RELEASE);
            } else com_err_set(&err_inf,"Unable to allocate %ld bytes of memory.\n",disk_file_size);
          } else com_err_set(&err_inf,"Unable to get size of file \"%s\".\n",used_path);

          CloseHandle(disk_file);
        } else
        {
          if (used_path==name) com_err_set(&err_inf,"Unable to open file \"%s\", try to copy it to this test's directory and try the test again.\n",full_path);
          else com_err_set(&err_inf,"Unable to open file \"%s\".\n",full_path);
          break;
        }
      }

      if (com_verbosity_get()) fprintf(conout,"\n");
    }
  }

  if (err_inf.occurred) com_err_print(&err_inf,COM_ERR_STANDARD_PREFIX);
  else res=TRUE;

  return res;
}
